/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.trendz.task;

import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.thingsboard.trendz.dao.ml.TaskDao;
import org.thingsboard.trendz.dao.sql.TimeStampUUIDGenerator;
import org.thingsboard.trendz.task.Task;
import org.thingsboard.trendz.task.TaskProgress;
import org.thingsboard.trendz.task.TaskStatus;
import org.thingsboard.trendz.task.TaskType;

@Component
public class TaskService {
    private static final Logger log = LoggerFactory.getLogger(TaskService.class);
    @Autowired
    private TaskDao taskDao;
    private ExecutorService taskExecutorService;
    private ScheduledExecutorService taskCacheClearExecutor;
    @Value(value="${executors.taskService}")
    private int concurrentTaskThreadCount;
    private Map<UUID, Task> taskCache = new ConcurrentHashMap();

    @PostConstruct
    public void init() {
        this.taskExecutorService = Executors.newFixedThreadPool(this.concurrentTaskThreadCount);
        this.taskCacheClearExecutor = Executors.newSingleThreadScheduledExecutor();
        this.finishHangedTasks();
    }

    @PreDestroy
    public void close() {
        if (this.taskExecutorService != null) {
            this.taskExecutorService.shutdownNow();
        }
        if (this.taskCacheClearExecutor != null) {
            this.taskCacheClearExecutor.shutdownNow();
        }
    }

    public <T> Task<T> submitTask(TaskType type, UUID modelId, Function<Task, T> job) {
        Task task = this.createTask(type, modelId);
        Future<?> future = this.taskExecutorService.submit(() -> {
            this.onTaskStarted(task);
            try {
                Object result = job.apply(task);
                this.onTaskReady(task, result);
            }
            catch (Throwable th) {
                this.onTaskCanceled(task, th.getMessage());
                log.error("Task {} failed", (Object)task.getId(), (Object)th);
            }
        });
        task.setFutureJob(future);
        return task;
    }

    public void cancelTask(UUID taskId) {
        Task task = this.getTask(taskId);
        if (task == null) {
            log.warn("Could not cancel task that does nto exist {}", (Object)taskId);
        } else if (task.getFutureJob().isDone()) {
            log.warn("Task {} already finished. Skip cancel", (Object)taskId);
        } else {
            try {
                boolean cancel = task.getFutureJob().cancel(true);
                log.info("Task {} canceled - {}", (Object)taskId, (Object)cancel);
            }
            catch (Throwable th) {
                log.error("could not cancel task", th);
            }
        }
    }

    public Task getTask(UUID taskId) {
        Task task = (Task)this.taskCache.get(taskId);
        if (task == null) {
            task = this.taskDao.findById(taskId);
        }
        if (task == null) {
            log.warn("Task {} does not exist", (Object)taskId);
            throw new IllegalStateException("Task not found with ID " + taskId);
        }
        return task;
    }

    public List<Task> findAnomalyTasksForModel(UUID modelId) {
        return this.taskDao.findModelTasks(modelId);
    }

    public void removeTaskByModelId(UUID modelId) {
        this.taskDao.removeModelTask(modelId);
    }

    public boolean removeTaskById(UUID taskId) {
        this.taskDao.delete(taskId);
        this.taskCache.remove(taskId);
        return true;
    }

    private Task createTask(TaskType type, UUID modelId) {
        Task task = new Task();
        task.setId(TimeStampUUIDGenerator.generateId());
        task.setModelId(modelId);
        task.setType(type);
        task.setCreatedAt(System.currentTimeMillis());
        task.setStatus(TaskStatus.QUEUED);
        task.setProgress(new TaskProgress());
        task = this.taskDao.save(task);
        this.taskCache.put(task.getId(), task);
        return task;
    }

    private <T> void onTaskStarted(Task<T> task) {
        task.setStartedAt(System.currentTimeMillis());
        task.setStatus(TaskStatus.IN_PROGRESS);
        task = this.taskDao.save(task);
    }

    private <T> void onTaskReady(Task<T> task, T result) {
        task.setResult(result);
        task.setStatus(TaskStatus.READY);
        task.setCompletedAt(System.currentTimeMillis());
        task = this.taskDao.save(task);
        this.scheduleCacheClear(task.getId());
        log.info("Task is Ready, id = {}", (Object)task.getId());
    }

    private <T> void onTaskCanceled(Task<T> task, String reason) {
        task.setCompletedAt(System.currentTimeMillis());
        task.setCompleteReason(reason != null && reason.length() > 1000 ? reason.substring(1000) : reason);
        task.setStatus(TaskStatus.CANCELED);
        task = this.taskDao.save(task);
        this.scheduleCacheClear(task.getId());
    }

    private void finishHangedTasks() {
        List inProgressTasks = this.taskDao.findTasksInStatus(TaskStatus.IN_PROGRESS);
        log.info("Found {} hanged tasks during server start", (Object)inProgressTasks.size());
        inProgressTasks.forEach(task -> this.onTaskCanceled(task, "Clear hanged task"));
        log.info("Hanged task clear finished");
    }

    private void scheduleCacheClear(UUID taskId) {
        this.taskCacheClearExecutor.schedule(() -> (Task)this.taskCache.remove(taskId), 20L, TimeUnit.SECONDS);
    }
}

