/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.trendz.service.provider.ws;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.net.URI;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.java_websocket.client.WebSocketClient;
import org.java_websocket.handshake.ServerHandshake;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.thingsboard.trendz.service.provider.ws.WsTask;
import org.thingsboard.trendz.service.provider.ws.domain.EntityDataUpdate;
import reactor.core.publisher.Mono;

public class ReactiveWsClient
extends WebSocketClient {
    private static final Logger log = LoggerFactory.getLogger(ReactiveWsClient.class);
    private static final long wsTaskTimeoutMs = TimeUnit.SECONDS.toMillis(60L);
    private static final ObjectMapper mapper = new ObjectMapper();
    private Map<Integer, WsTask> cmdSinkMap = new ConcurrentHashMap();
    private final AtomicBoolean isActive = new AtomicBoolean(false);
    private final String wsUrl;
    private ScheduledExecutorService taskCheckExecutor;

    public ReactiveWsClient(String wsUrl, URI uri, String token) {
        super(uri);
        this.wsUrl = wsUrl;
        this.connectToServer(token);
        this.taskCheckExecutor = Executors.newSingleThreadScheduledExecutor();
        this.taskCheckExecutor.scheduleAtFixedRate(() -> this.clearExpiredTasks(), 10L, 10L, TimeUnit.SECONDS);
        log.info("ReactiveWsClient created");
    }

    private synchronized void connectToServer(String token) {
        try {
            if (this.isActive.get()) {
                return;
            }
            this.uri = new URI(this.wsUrl + "?token=" + token);
            if (!this.connectBlocking()) {
                this.isActive.set(false);
                throw new IllegalStateException("Could not create WS connection");
            }
            this.isActive.set(true);
        }
        catch (Exception ex) {
            this.isActive.set(false);
            log.error("Error while create WS connection", (Throwable)ex);
            throw new IllegalStateException("Error while create WS connection", ex);
        }
    }

    public void close() {
        log.warn("Closing Ws Client");
        if (this.taskCheckExecutor != null) {
            this.taskCheckExecutor.shutdownNow();
        }
        for (WsTask wsTask : this.cmdSinkMap.values()) {
            wsTask.getSink().error((Throwable)new IllegalStateException("Ws connection closed"));
        }
        super.close();
    }

    public Mono<EntityDataUpdate> submitCommand(int cmdId, Object payload, String token) {
        if (!this.isActive.get()) {
            this.connectToServer(token);
        }
        String raw = this.convertToText(payload);
        log.trace("send command {}", (Object)raw);
        Mono mono = Mono.create(sink -> {
            this.cmdSinkMap.put(cmdId, new WsTask(sink, System.currentTimeMillis()));
            this.send(raw);
        });
        return mono;
    }

    private EntityDataUpdate parseDataUpdate(String raw) {
        try {
            return (EntityDataUpdate)mapper.readValue(raw, EntityDataUpdate.class);
        }
        catch (IOException e) {
            throw new IllegalStateException("Could not parse WS responce " + raw, e);
        }
    }

    private String convertToText(Object obj) {
        try {
            return mapper.writeValueAsString(obj);
        }
        catch (JsonProcessingException e) {
            throw new IllegalStateException(e);
        }
    }

    public void onOpen(ServerHandshake handshakedata) {
        log.trace("Ws client session opened {}", (Object)handshakedata);
        this.isActive.set(true);
    }

    public void onMessage(String message) {
        log.trace("WS message received {}", (Object)message);
        EntityDataUpdate dataUpdate = this.parseDataUpdate(message);
        WsTask wsTask = (WsTask)this.cmdSinkMap.get(dataUpdate.getCmdId());
        if (wsTask != null) {
            this.cmdSinkMap.remove(dataUpdate.getCmdId());
            wsTask.getSink().success((Object)dataUpdate);
        } else {
            log.warn("No destination found for incoming message {}", (Object)message);
        }
    }

    public void onClose(int code, String reason, boolean remote) {
        this.isActive.set(false);
        this.close();
    }

    public void onError(Exception ex) {
        log.error("Ws client session error", (Throwable)ex);
        this.isActive.set(false);
        this.close();
    }

    private void clearExpiredTasks() {
        Set<Integer> expiredTaskIds = this.cmdSinkMap.entrySet().stream().filter(t -> System.currentTimeMillis() - ((WsTask)t.getValue()).getStartTs() > wsTaskTimeoutMs).map(Map.Entry::getKey).collect(Collectors.toSet());
        expiredTaskIds.forEach(cmdId -> {
            WsTask task = (WsTask)this.cmdSinkMap.remove(cmdId);
            if (task != null) {
                task.getSink().error((Throwable)new IllegalStateException("ws task expired"));
            }
        });
    }
}

