CREATE TABLE IF NOT EXISTS metric_definition (
    id                      UUID NOT NULL CONSTRAINT metric_definition_pkey PRIMARY KEY,

    business_entity_id      UUID NOT NULL,
    item_id                 UUID NOT NULL,
    item_name               VARCHAR(255) NOT NULL,

    tenant_id               UUID NOT NULL,
    customer_id             UUID,

    name                    TEXT NOT NULL,
    user_input              TEXT NOT NULL,
    description             TEXT NOT NULL,
    how_to_calculate        TEXT NOT NULL,
    is_advanced_mode        BOOLEAN NOT NULL DEFAULT FALSE,

    use_cases               TEXT,
    fields                  TEXT,
    code                    TEXT,
    calculation_field_id    UUID,
    is_outdated             BOOLEAN NOT NULL DEFAULT FALSE,

    auto_deletable          BOOLEAN NOT NULL DEFAULT TRUE,

    created_ts              BIGINT NOT NULL,
    updated_ts              BIGINT NOT NULL
);

CREATE TABLE IF NOT EXISTS metric_definition_metadata (
    id                      UUID NOT NULL CONSTRAINT metric_definition_metadata_pkey PRIMARY KEY,

    metric_definition_id    UUID NOT NULL,
    business_entity_id      UUID NOT NULL,
    item_id                 UUID NOT NULL,

    tenant_id               UUID NOT NULL,
    customer_id             UUID,

    metric_name             TEXT NOT NULL,
    is_advanced_mode        BOOLEAN NOT NULL DEFAULT FALSE,

    is_saved                BOOLEAN NOT NULL,
    created_ts              BIGINT NOT NULL
);

ALTER TABLE view_field
ADD COLUMN IF NOT EXISTS native_calculation BOOLEAN NOT NULL DEFAULT FALSE;

UPDATE business_entity_field
SET type = 'STRING'
WHERE (query::jsonb ->> 'queryType') IN ('ENTITY_NAME', 'ENTITY_ID', 'ENTITY_LABEL', 'OWNER')
    AND type IS DISTINCT FROM 'STRING';

ALTER TABLE view_assistance_chat
ADD COLUMN IF NOT EXISTS type VARCHAR(255) NOT NULL DEFAULT 'VIEW_ASSISTANT';

ALTER TABLE view_assistance_chat
ADD COLUMN IF NOT EXISTS metric_definition_id UUID;

ALTER TABLE view_assistance_chat_message
ADD COLUMN IF NOT EXISTS code TEXT;

UPDATE trendz_task
SET job_type = 'CHAT_ASSISTANCE'
WHERE job_type = 'VIEW_REPORT_ASSISTANCE';

UPDATE trendz_task
SET reference_type = 'CHAT_MESSAGE_PROCESS_TASK'
WHERE reference_type = 'VIEW_ASSISTANCE_CHAT_MESSAGE_PROCESS_TASK';

UPDATE trendz_task_execution
SET job_type = 'CHAT_ASSISTANCE'
WHERE job_type = 'VIEW_REPORT_ASSISTANCE';

INSERT INTO trendz_task (
    id,
    tenant_id,
    customer_id,
    user_id,
    created_ts,
    updated_ts,
    name,
    enabled,
    reference_type,
    reference_key,
    job_type,
    json_job,
    schedule_type,
    schedule_period_ts,
    schedule_planned_ts,
    schedule_scheduling_unit,
    schedule_scheduling_unit_count,
    schedule_scheduling_time_zone,
    ttl_enabled,
    ttl_duration,
    store_execution_enabled,
    store_execution_count,
    json_configs
)
SELECT
    gen_random_uuid()                                                                       AS id,
    t1.tenant_id                                                                            AS tenant_id,
    t1.customer_id                                                                          AS customer_id,
    t1.user_id                                                                              AS user_id,
    t1.created_ts                                                                           AS created_ts,
    t1.updated_ts                                                                           AS updated_ts,
    regexp_replace(t1.name, '^View config, build:', 'View config, AI summary:')             AS name,
    t1.enabled                                                                              AS enabled,
    'VIEW_CONFIG_AI_SUMMARY_TASK'                                                           AS reference_type,
    t1.reference_key                                                                        AS reference_key,
    'AI_SUMMARY_REQUEST'                                                                    AS job_type,
    '{}'                                                                                    AS json_job,
    'NONE'                                                                                  AS schedule_type,
    0                                                                                       AS schedule_period_ts,
    0                                                                                       AS schedule_planned_ts,
    'HOURS'                                                                                 AS schedule_scheduling_unit,
    1                                                                                       AS schedule_scheduling_unit_count,
    'UTC'                                                                                   AS schedule_scheduling_time_zone,
    TRUE                                                                                    AS ttl_enabled,
    259200000                                                                               AS ttl_duration,
    TRUE                                                                                    AS store_execution_enabled,
    10                                                                                      AS store_execution_count,
    '{"timeoutConfig":{"enabled":true,"duration":3600000},"retryPolicy":{"type":"NONE"}}'   AS json_configs
FROM trendz_task AS t1
WHERE job_type = 'VIEW_REPORT_BUILD'
    AND NOT EXISTS (
        SELECT 1
        FROM trendz_task AS t2
        WHERE t2.reference_key = t1.reference_key
            AND t2.reference_type = 'VIEW_CONFIG_AI_SUMMARY_TASK'
    );

INSERT INTO trendz_task (
    id,
    tenant_id,
    customer_id,
    user_id,
    created_ts,
    updated_ts,
    name,
    enabled,
    reference_type,
    reference_key,
    job_type,
    json_job,
    schedule_type,
    schedule_period_ts,
    schedule_planned_ts,
    schedule_scheduling_unit,
    schedule_scheduling_unit_count,
    schedule_scheduling_time_zone,
    ttl_enabled,
    ttl_duration,
    store_execution_enabled,
    store_execution_count,
    json_configs
)
SELECT
    gen_random_uuid()                                                                       AS id,
    cp.tenant_id                                                                            AS tenant_id,
    COALESCE(cp.customer_id, '13814000-1dd2-11b2-8080-808080808080'::uuid)                  AS customer_id,
    cp.user_id                                                                              AS user_id,
    cp.created_ts                                                                           AS created_ts,
    cp.last_modified_ts                                                                     AS updated_ts,
    concat('AI prompt execution: ', cp.name)                                                AS name,
    true                                                                                    AS enabled,
    'AI_PROMPT_EXECUTION_TASK'                                                              AS reference_type,
    cp.id::text                                                                             AS reference_key,
    'AI_PROMPT_EXECUTION'                                                                   AS job_type,
    '{}'                                                                                    AS json_job,
    'NONE'                                                                                  AS schedule_type,
    0                                                                                       AS schedule_period_ts,
    0                                                                                       AS schedule_planned_ts,
    'HOURS'                                                                                 AS schedule_scheduling_unit,
    1                                                                                       AS schedule_scheduling_unit_count,
    'UTC'                                                                                   AS schedule_scheduling_time_zone,
    TRUE                                                                                    AS ttl_enabled,
    259200000                                                                               AS ttl_duration,
    TRUE                                                                                    AS store_execution_enabled,
    10                                                                                      AS store_execution_count,
    '{"timeoutConfig":{"enabled":true,"duration":3600000},"retryPolicy":{"type":"NONE"}}'   AS json_configs
FROM custom_prompt cp
WHERE NOT cp.is_system
    AND NOT cp.is_deleted
    AND NOT EXISTS (
        SELECT 1
        FROM trendz_task AS tt
        WHERE cp.id::text = tt.reference_key
        AND tt.reference_type = 'AI_PROMPT_EXECUTION_TASK'
    );

DELETE FROM trendz_task
WHERE job_type = 'VIEW_AI_SUMMARY_EXECUTION'
    OR job_type = 'CUSTOM_PROMPT_EXECUTION';

DELETE FROM trendz_task_execution
WHERE job_type = 'VIEW_AI_SUMMARY_EXECUTION'
    OR job_type = 'CUSTOM_PROMPT_EXECUTION';

ALTER TABLE datasource
ADD COLUMN IF NOT EXISTS db_type VARCHAR(255) NOT NULL DEFAULT 'POSTGRESQL';



----
CREATE TABLE IF NOT EXISTS llm_config (
    id                  UUID NOT NULL CONSTRAINT llm_config_pkey PRIMARY KEY,
    tenant_id           UUID NOT NULL,
    created_ts          BIGINT NOT NULL,
    updated_ts          BIGINT NOT NULL,
    name                VARCHAR(255) NOT NULL,
    model_provider      VARCHAR(255) NOT NULL,
    model_name          VARCHAR(255) NOT NULL,
    json_properties     VARCHAR(10240) NOT NULL
);

CREATE TABLE IF NOT EXISTS llm_settings (
     id                     UUID NOT NULL CONSTRAINT llm_settings_pkey PRIMARY KEY,
     tenant_id              UUID NOT NULL,
     active                 BOOLEAN NOT NULL,
     use_default            BOOLEAN NOT NULL,
     default_llm_config_id  UUID,

    UNIQUE (tenant_id),
    FOREIGN KEY (default_llm_config_id) REFERENCES llm_config (id) ON DELETE SET NULL
);

CREATE TABLE IF NOT EXISTS llm_settings_chat_type_link (
     llm_setting_id     UUID NOT NULL,
     chat_type          VARCHAR(255) NOT NULL,
     llm_config_id      UUID NOT NULL,

    PRIMARY KEY (llm_setting_id, chat_type),
    FOREIGN KEY (llm_setting_id) REFERENCES llm_settings (id) ON DELETE CASCADE,
    FOREIGN KEY (llm_config_id) REFERENCES llm_config (id) ON DELETE CASCADE
);

CREATE INDEX IF NOT EXISTS llm_config_tenant_id_idx                                 ON llm_config (tenant_id);
CREATE INDEX IF NOT EXISTS llm_config_created_ts_idx                                ON llm_config (created_ts);
CREATE INDEX IF NOT EXISTS llm_config_updated_ts_idx                                ON llm_config (updated_ts);
CREATE INDEX IF NOT EXISTS llm_settings_chat_type_link_llm_config_id_idx            ON llm_settings_chat_type_link (llm_config_id);
CREATE INDEX IF NOT EXISTS llm_settings_chat_type_link_setting_id_idx               ON llm_settings_chat_type_link (llm_setting_id);
---
-- Перенос из старых view_* таблиц в новые, с преобразованием json_properties
DO $$
BEGIN
    IF to_regclass('public.view_assistance_model_config') IS NOT NULL THEN
        WITH ts AS (
            SELECT (extract(epoch FROM now()) * 1000)::bigint AS now_ms
        )
        INSERT INTO llm_config (
            id,
            tenant_id,
            created_ts,
            updated_ts,
            name,
            model_provider,
            model_name,
            json_properties
        )
        SELECT
            vmc.id,
            vmc.tenant_id,
            ts.now_ms,
            ts.now_ms,
            COALESCE(NULLIF(vmc.model_name, ''), vmc.model_provider) AS name,
            vmc.model_provider,
            COALESCE(NULLIF(vmc.model_name, ''), vmc.model_provider) AS model_name,
            (
                jsonb_build_object(
                    'temperature', 0.0,
                    'topP', 0.95,
                    'providerProperties',
                        COALESCE(NULLIF(vmc.json_properties, '')::jsonb, '{}'::jsonb)
                )
            )::text
        FROM view_assistance_model_config vmc
        CROSS JOIN ts
        ON CONFLICT (id) DO NOTHING;
    END IF;
END $$;

DO $$
    BEGIN
        IF to_regclass('public.view_assistance_admin_config') IS NOT NULL THEN
            INSERT INTO llm_settings (
                id, tenant_id, active, use_default, default_llm_config_id
            )
            SELECT
                vac.id,
                vac.tenant_id,
                true,
                true,
                vac.main_model_config
            FROM view_assistance_admin_config vac
            WHERE vac.main_model_config IS NOT NULL
            ON CONFLICT (tenant_id) DO UPDATE
                SET default_llm_config_id = EXCLUDED.default_llm_config_id,
                    active = llm_settings.active OR (EXCLUDED.default_llm_config_id IS NOT NULL);
        END IF;

        UPDATE llm_settings
        SET active = TRUE
        WHERE active IS NOT TRUE AND default_llm_config_id IS NOT NULL;
    END
$$;

---

CREATE TABLE IF NOT EXISTS business_entity_metadata (
    id                          UUID NOT NULL CONSTRAINT business_entity_metadata_pkey PRIMARY KEY,

    business_entity_id          UUID NOT NULL,

    tenant_id                   UUID NOT NULL,
    customer_id                 UUID NOT NULL,

    item_id                     UUID,
    item_name                   VARCHAR(256),

    CONSTRAINT business_entity_metadata_unique UNIQUE (business_entity_id, tenant_id, customer_id)
);

CREATE TABLE IF NOT EXISTS business_entity_field_metadata (
    id                          UUID NOT NULL CONSTRAINT business_entity_field_metadata_pkey PRIMARY KEY,

    business_entity_field_id    UUID NOT NULL,

    tenant_id                   UUID NOT NULL,
    customer_id                 UUID NOT NULL,

    aggregation                 VARCHAR(32),
    date_grouping               VARCHAR(32),
    range_config                TEXT,

    CONSTRAINT business_entity_field_metadata_unique UNIQUE (business_entity_field_id, tenant_id, customer_id)
);

CREATE TABLE IF NOT EXISTS user_metadata (
    id                          UUID NOT NULL CONSTRAINT user_metadata_pkey PRIMARY KEY,

    tenant_id                   UUID NOT NULL,
    customer_id                 UUID NOT NULL,

    default_business_entity_id  UUID,
    default_date_picker_config  TEXT,
    default_item_id             UUID,
    default_item_name           VARCHAR(256),

    CONSTRAINT user_metadata_unique UNIQUE (tenant_id, customer_id)
);

ALTER TABLE business_entity ADD COLUMN IF NOT EXISTS shared_with_customers BOOLEAN NOT NULL DEFAULT false;
