export function getAnalytics(widgetCtx, isDefault, prompt) {
    if (!isDefault && !prompt) {
        throw new Error('You should indicate prompt/promptId or use default prompt');
    }

    const services = initializeServices(widgetCtx);
    const dialogConfig = createDialogConfig();

    injectStyles(DIALOG_STYLES);

    services.customDialog.customDialog(createDialogTemplate(), createDialogController(services, isDefault, prompt), null, dialogConfig).subscribe();
}

function initializeServices(widgetCtx) {
    const ctx = widgetCtx;
    const $injector = ctx.$scope.$injector;

    return {
        ctx,
        customDialog: $injector.get(ctx.servicesMap.get('customDialog')),
        widgetService: $injector.get(ctx.servicesMap.get('entityService')).widgetService,
        jwtToken: localStorage.getItem('jwt_token'),
        rxjs: ctx.rxjs,
        title: getWidgetTitle(widgetCtx)
    };
}

function getWidgetTitle(widgetCtx) {
    const config = widgetCtx.widget.config;
    return config.showTitle && config.title ? config.title : '';
}

function createDialogConfig() {
    return {
        width: '600px',
        maxWidth: '90%'
    };
}

function createDialogTemplate() {
    return `
        <div *ngIf="title" mat-dialog-title>{{title}}</div>
        <div mat-dialog-content style="padding: 0 24px">
            <div *ngIf="!summaryContent" class="flex items-center justify-center">
                <mat-spinner [diameter]="40"></mat-spinner>
            </div>
            <div *ngIf="summaryContent" class="tz-markdown-content" [innerHTML]="summaryContent"></div>
        </div>
        <div mat-dialog-actions class="flex items-center justify-end">
            <button mat-raised-button color="primary" (click)="cancel()">Close</button>
        </div>`;
}

function createDialogController(services, isDefault, prompt) {
    return function(instance) {
        const controller = new AnalyticsDialogController(instance, services, isDefault, prompt);
        controller.initialize();
    };
}

function injectStyles(styleContent) {
    const styleEl = document.createElement('style');
    styleEl.type = 'text/css';
    styleEl.setAttribute('id', 'custom-tz-dialog-styles');
    styleEl.appendChild(document.createTextNode(styleContent));
    document.head.appendChild(styleEl);
}

function removeStyles() {
    const styleEl = document.getElementById('custom-tz-dialog-styles');
    if (styleEl) {
        document.head.removeChild(styleEl);
    }
}

class AnalyticsDialogController {
    constructor(instance, services, isDefault, prompt) {
        this.instance = instance;
        this.services = services;
        this.isDefault = isDefault;
        this.prompt = prompt;
    }

    initialize() {
        this.setupInstance();
        this.loadAnalytics();
    }

    setupInstance() {
        const { instance, services } = this;

        instance.summaryContent = '';
        instance.title = services.title;
        instance.cancel = () => {
            removeStyles();
            instance.dialogRef.close(null)

        };
    }

    loadAnalytics() {
        const { services, isDefault, prompt } = this;
        const dataForExport = services.ctx.defaultSubscription.exportData();
        const summaryData = DataTransformer.toCsv(dataForExport);

        AnalyticsUrlService.getUrl(services.widgetService, services.ctx, (url, error) => {
            if (error || !url) {
                console.error('Analytics URL not found.', error);
                this.instance.dialogRef.close(null);
                return;
            }

            this.executeSummaryRequest(summaryData, url);
        });
    }

    executeSummaryRequest(summaryData, url) {
        const { services, isDefault, prompt } = this;
        const { delay } = services.rxjs;

        AnalyticsApiService.getSummary(services.ctx, isDefault, prompt, summaryData, services.jwtToken, url)
            .pipe(delay(200))
            .subscribe({
                next: (execution) => this.handleSummaryResponse(execution, url),
                error: (error) => this.handleError('Error fetching summary:', error)
            });
    }

    handleSummaryResponse(execution, url) {
        const { services } = this;

        TaskExecutionService.fetchTask(services.ctx, execution, services.jwtToken, url)
            .subscribe(res => {
                this.instance.summaryContent = MarkdownParser.parse(res) || 'Summary wasn`t generated!';
            });
    }

    handleError(message, error) {
        console.error(message, error);
        this.instance.dialogRef.close(null);
    }
}

class DataTransformer {
    static toCsv(data) {
        if (!data || !data.length) {
            return '';
        }

        const header = Object.keys(data[0]).join(';');
        const rows = data.map(obj => Object.values(obj).join(';')).join('\n');
        return `${header}\n${rows}`;
    }
}

class AnalyticsApiService {
    static getSummary(ctx, isDefault, prompt, summaryData, jwtToken, analyticsLink) {
        const summaryRequest = this.buildSummaryRequest(isDefault, prompt, summaryData);

        return ctx.http.post(`${analyticsLink}/apiTrendz/agent/prompts/execute`, summaryRequest, {
            headers: { Jwt: jwtToken }
        });
    }

    static buildSummaryRequest(isDefault, prompt, summaryData) {
        let summaryRequest = { data: summaryData };

        if (!isDefault && prompt) {
            summaryRequest = ValidationUtils.isValidUUID(prompt)
                ? { promptId: prompt, data: summaryData }
                : { prompt: prompt, data: summaryData };
        }

        return summaryRequest;
    }
}

class TaskExecutionService {
    static fetchTask(ctx, executionId, jwtToken, analyticsLink) {
        return this.pollExecution(ctx, executionId, jwtToken, analyticsLink)
            .pipe(ctx.rxjs.switchMap(res => this.handleExecutionResult(ctx, res, executionId, jwtToken, analyticsLink)));
    }

    static handleExecutionResult(ctx, res, executionId, jwtToken, analyticsLink) {
        if (res.success) {
            return this.handleSuccessResult(ctx, res.success, executionId, jwtToken, analyticsLink);
        } else if (res.canceled) {
            const errorMessage = ErrorHandler.buildErrorMessage(res.canceled);
            return ctx.rxjs.of(errorMessage);
        }
    }

    static handleSuccessResult(ctx, success, executionId, jwtToken, analyticsLink) {
        if (success.status !== 'FINISHED') {
            return this.fetchTask(ctx, executionId, jwtToken, analyticsLink);
        } else {
            return ctx.rxjs.of(success.jsonResult ? success.jsonResult.result : null);
        }
    }

    static pollExecution(ctx, executionId, jwtToken, analyticsLink) {
        return ctx.http.get(`${analyticsLink}/apiTrendz/task/execution/poll/${executionId}`, {
            headers: { Jwt: jwtToken }
        }).pipe(
            ctx.rxjs.delay(300),
            ctx.rxjs.switchMap(execution => this.processExecutionResponse(ctx, execution))
        );
    }

    static processExecutionResponse(ctx, execution) {
        if (!execution) {
            return ctx.rxjs.of({ canceled: 'Failed to execute task' });
        }

        const validStatuses = ['CREATED', 'FINISHED', 'RUNNING'];
        if (validStatuses.includes(execution.status)) {
            return ctx.rxjs.of({ success: execution });
        }

        return ctx.rxjs.of({ canceled: execution.jsonResult });
    }
}

class AnalyticsUrlService {
    static getUrl(widgetService, ctx, callback) {
        const pageLink = ctx.pageLink(100, 0, 'analysis_results');

        widgetService.getWidgetBundles(pageLink, true).subscribe(
            (res) => this.handleBundlesResponse(res, widgetService, callback),
            (error) => callback(null, error.message || error)
        );
    }

    static handleBundlesResponse(res, widgetService, callback) {
        if (!res || !res.data || !res.data.length) {
            callback(null, 'Analytics bundle not found');
            return;
        }

        const analyticsBundle = res.data.find(bundle => bundle.alias === 'trendz_bundle');
        if (!analyticsBundle) {
            callback(null, 'Analytics bundle not found');
            return;
        }

        this.getWidgetDetails(widgetService, analyticsBundle.id.id, callback);
    }

    static getWidgetDetails(widgetService, analyticsBundleId, callback) {
        widgetService.exportBundleWidgetTypesDetails(analyticsBundleId, true).subscribe(
            (widgets) => this.handleWidgetsResponse(widgets, callback),
            (error) => callback(null, error.message || error)
        );
    }

    static handleWidgetsResponse(widgets, callback) {
        const analyticsWidget = widgets.find(wdg => wdg.fqn === 'trendz_bundle.trendz_view_latest');

        if (!analyticsWidget || !analyticsWidget.descriptor || !analyticsWidget.descriptor.resources.length) {
            callback(null, 'Analytics widget not found');
            return;
        }

        const libUrl = analyticsWidget.descriptor.resources[0].url;
        const url = new URL(libUrl).origin;
        callback(url, null);
    }
}

class ValidationUtils {
    static isValidUUID(uuid) {
        const uuidRegex = /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;
        return uuidRegex.test(uuid);
    }
}

class ErrorHandler {
    static buildErrorMessage(errRes) {
        try {
            return errRes.map((el, idx) => {
                const message = this.extractErrorMessage(el);
                return idx === errRes.length - 1 ? message : message + '\n';
            }).join('');
        } catch (e) {
            console.log(e);
            return 'Invalid response!';
        }
    }

    static extractErrorMessage(errorElement) {
        if (!errorElement.message) {
            return '';
        }

        if (this.isJsonString(errorElement.message)) {
            return JSON.parse(errorElement.message).message;
        }

        return errorElement.message;
    }

    static isJsonString(str) {
        try {
            JSON.parse(str);
            return true;
        } catch {
            return false;
        }
    }
}

class MarkdownParser {
    static parse(content) {
        if (!content) return null;

        let summary = this.escapeHtml(content);

        summary = this.parseCodeBlocks(summary);
        summary = this.parseBlockquotes(summary);
        summary = this.processLists(summary);
        summary = this.parseHeaders(summary);
        summary = this.parseHorizontalRules(summary);

        summary = this.parseInlineCode(summary);
        summary = this.parseBoldText(summary);
        summary = this.parseItalicText(summary);
        summary = this.parseLinks(summary);

        summary = this.parseParagraphs(summary);

        return summary;
    }

    static escapeHtml(content) {
        return content
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;');
    }

    static parseHeaders(summary) {
        return summary
            .replace(/^###### (.*$)/gm, '<h6>$1</h6>')
            .replace(/^##### (.*$)/gm, '<h5>$1</h5>')
            .replace(/^#### (.*$)/gm, '<h4>$1</h4>')
            .replace(/^### (.*$)/gm, '<h3>$1</h3>')
            .replace(/^## (.*$)/gm, '<h2>$1</h2>')
            .replace(/^# (.*$)/gm, '<h1>$1</h1>');
    }

    static parseHorizontalRules(summary) {
        return summary.replace(/^---$/gm, '<hr>');
    }

    static parseBlockquotes(summary) {
        return summary.replace(/^> (.*)$/gm, '<blockquote>$1</blockquote>');
    }

    static parseCodeBlocks(summary) {
        return summary.replace(/```([\s\S]*?)```/g, '<pre><code>$1</code></pre>');
    }

    static parseInlineCode(summary) {
        return summary.replace(/`([^`]+)`/g, '<code>$1</code>');
    }

    static parseBoldText(summary) {
        return summary
            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
            .replace(/__(.*?)__/g, '<strong>$1</strong>');
    }

    static parseItalicText(summary) {
        return summary
            .replace(/\*(.*?)\*/g, '<em>$1</em>')
            .replace(/_(.*?)_/g, '<em>$1</em>');
    }

    static parseLinks(summary) {
        return summary.replace(/\[([^\]]+)\]\(([^)]+)\)/g,
            '<a href="$2" target="_blank" rel="noopener noreferrer">$1</a>');
    }

    static processLists(summary) {
        const lines = summary.split('\n');
        const processor = new ListProcessor();
        return processor.process(lines);
    }

    static parseParagraphs(summary) {
        const lines = summary.split('\n');
        let result = '';
        let currentParagraph = [];
        for (const line of lines) {
            const isBlockElement = line.match(/^<(h[1-6]|ul|ol|blockquote|pre|hr|div)/);

            if (isBlockElement) {
                if (currentParagraph.length > 0) {
                    result += `<p>${currentParagraph.join('<br>')}</p>`;
                    currentParagraph = [];
                }
                result += line;
            } else if (line.trim() === '') {
                if (currentParagraph.length > 0) {
                    result += `<p>${currentParagraph.join('<br>')}</p>`;
                    currentParagraph = [];
                }
            } else {
                currentParagraph.push(line);
            }
        }

        if (currentParagraph.length > 0) {
            result += `<p>${currentParagraph.join('<br>')}</p>`;
        }

        result = result.replace(/<\/p><p>/g, '');

        return result.trim();
    }
}

class ListProcessor {
    constructor() {
        this.reset();
    }

    reset() {
        this.result = [];
        this.listStack = [];
    }

    process(lines) {
        this.reset();

        for (const line of lines) {
            this.processLine(line);
        }

        this.closeAllLists();
        return this.result.join('\n');
    }

    processLine(line) {
        const listMatch = this.parseListItem(line);

        if (listMatch) {
            this.processListItem(listMatch);
        } else {
            this.processNonListLine(line);
        }
    }

    parseListItem(line) {
        const unorderedMatch = line.match(/^(\s*)([*-]) (.*)$/);
        if (unorderedMatch) {
            return {
                indent: unorderedMatch[1].length,
                type: 'ul',
                content: unorderedMatch[3],
                marker: unorderedMatch[2]
            };
        }

        const orderedMatch = line.match(/^(\s*)(\d+)\. (.*)$/);
        if (orderedMatch) {
            return {
                indent: orderedMatch[1].length,
                type: 'ol',
                content: orderedMatch[3],
                marker: orderedMatch[2]
            };
        }

        return null;
    }

    processListItem(listMatch) {
        const { indent, type, content } = listMatch;
        const level = Math.floor(indent / 4);

        this.adjustListStack(level, type);

        if (this.listStack.length > 0) {
            const currentList = this.listStack[this.listStack.length - 1];
            currentList.items.push(`<li>${content}</li>`);
        }
    }

    adjustListStack(targetLevel, listType) {
        while (this.listStack.length > targetLevel + 1) {
            this.closeCurrentList();
        }

        if (this.listStack.length === targetLevel + 1) {
            const currentList = this.listStack[this.listStack.length - 1];
            if (currentList.type !== listType) {
                this.closeCurrentList();
                this.startNewList(listType, targetLevel);
            }
        } else if (this.listStack.length === targetLevel) {
            this.startNewList(listType, targetLevel);
        }
    }

    startNewList(listType, level) {
        const newList = {
            type: listType,
            level: level,
            items: []
        };
        this.listStack.push(newList);
    }

    closeCurrentList() {
        if (this.listStack.length === 0) return;

        const currentList = this.listStack.pop();
        const listHtml = `<${currentList.type}>${currentList.items.join('')}</${currentList.type}>`;

        if (this.listStack.length > 0) {
            const parentList = this.listStack[this.listStack.length - 1];
            if (parentList.items.length > 0) {
                const lastItemIndex = parentList.items.length - 1;
                const lastItem = parentList.items[lastItemIndex];
                parentList.items[lastItemIndex] = lastItem.replace('</li>', listHtml + '</li>');
            }
        } else {
            this.result.push(listHtml);
        }
    }

    closeAllLists() {
        while (this.listStack.length > 0) {
            this.closeCurrentList();
        }
    }

    processNonListLine(line) {
        this.closeAllLists();
        this.result.push(line);
    }
}

const DIALOG_STYLES = `
    .tz-markdown-content h1 {
        margin: 0;
        font-size: 2rem;
        line-height: 2.1rem;
    }
    
    .tz-markdown-content h2 {
        margin: 0;
        font-size: 1.7rem;
        line-height: 1.8rem;
    }
    
    .tz-markdown-content h3 {
        margin: 0;
        font-size: 1.4rem;
        line-height: 1.5rem;
    }
    
    .tz-markdown-content h4 {
        margin: 0;
        font-size: 1.1rem;
        line-height: 1.2rem;
    }
    
    .tz-markdown-content h5 {
        margin: 0;
        font-size: 1rem;
        line-height: 1.1rem;
    }
    
    .tz-markdown-content h6 {
        margin: 0;
        font-size: 0.9rem;
        line-height: 0.8rem;
    }
    
    .tz-markdown-content p {
        margin: 0;
        font-size: 1rem;
        line-height: 1.1rem;
    }
`;
