CREATE TABLE IF NOT EXISTS anomaly_model_task_data (
    anomaly_model_id                    UUID NOT NULL CONSTRAINT anomaly_model_task_data_pkey PRIMARY KEY,

    enabled_refresh                     BOOLEAN NOT NULL DEFAULT FALSE,
    enabled_save_to_tb                  BOOLEAN NOT NULL DEFAULT FALSE,
    enabled_alarm_creation              BOOLEAN NOT NULL DEFAULT FALSE,

    item_set_json                       TEXT,
    refresh_time_unit                   VARCHAR(32),
    refresh_time_unit_count             INTEGER,
    refresh_time_unit_truncated         BOOLEAN
);

ALTER TABLE cluster_model ADD COLUMN IF NOT EXISTS tb_telemetry_key VARCHAR(64);
ALTER TABLE cluster_model ADD COLUMN IF NOT EXISTS score_associated_field_id UUID;
ALTER TABLE cluster_model ADD COLUMN IF NOT EXISTS score_index_associated_field_id UUID;
ALTER TABLE cluster_model ADD COLUMN IF NOT EXISTS telemetry_save_period_unit VARCHAR(32);

ALTER TABLE cluster_model ADD COLUMN IF NOT EXISTS alarm_type VARCHAR(255);
ALTER TABLE cluster_model ADD COLUMN IF NOT EXISTS alarm_warning_threshold INTEGER;
ALTER TABLE cluster_model ADD COLUMN IF NOT EXISTS alarm_minor_threshold INTEGER;
ALTER TABLE cluster_model ADD COLUMN IF NOT EXISTS alarm_major_threshold INTEGER;
ALTER TABLE cluster_model ADD COLUMN IF NOT EXISTS alarm_critical_threshold INTEGER;

ALTER TABLE cluster_model ADD COLUMN IF NOT EXISTS enabled_alarm_deletion BOOLEAN NOT NULL DEFAULT TRUE;

ALTER TABLE anomaly ADD COLUMN IF NOT EXISTS alarm_id UUID;

ALTER TABLE dataset_config ADD COLUMN IF NOT EXISTS item_set TEXT;

UPDATE trendz_task
SET job_type = CASE job_type
    WHEN 'BUILD_MODEL' THEN 'ANOMALY_MODEL_BUILD'
    WHEN 'FIND_ANOMALIES' THEN 'ANOMALY_MODEL_REPROCESS'
    WHEN 'ANOMALY_AUTODISCOVERY' THEN 'ANOMALY_MODEL_REFRESH'
    ELSE job_type
END,
reference_type = CASE reference_type
    WHEN 'ANOMALY_MODEL_BUILD_MODEL' THEN 'ANOMALY_MODEL_BUILD'
    WHEN 'ANOMALY_MODEL_FIND_ANOMALIES' THEN 'ANOMALY_MODEL_REPROCESS'
    WHEN 'ANOMALY_MODEL_ANOMALY_AUTODISCOVERY' THEN 'ANOMALY_MODEL_REFRESH'
    ELSE reference_type
END
WHERE job_type IN ('BUILD_MODEL', 'FIND_ANOMALIES', 'ANOMALY_AUTODISCOVERY')
   OR reference_type IN (
       'ANOMALY_MODEL_BUILD_MODEL',
       'ANOMALY_MODEL_FIND_ANOMALIES',
       'ANOMALY_MODEL_ANOMALY_AUTODISCOVERY'
   );

UPDATE trendz_task_execution
SET job_type = CASE job_type
    WHEN 'BUILD_MODEL' THEN 'ANOMALY_MODEL_BUILD'
    WHEN 'FIND_ANOMALIES' THEN 'ANOMALY_MODEL_REPROCESS'
    WHEN 'ANOMALY_AUTODISCOVERY' THEN 'ANOMALY_MODEL_REFRESH'
    ELSE job_type
END
WHERE job_type IN ('BUILD_MODEL', 'FIND_ANOMALIES', 'ANOMALY_AUTODISCOVERY');

UPDATE trendz_task_execution_request
SET job_type = CASE job_type
    WHEN 'BUILD_MODEL' THEN 'ANOMALY_MODEL_BUILD'
    WHEN 'FIND_ANOMALIES' THEN 'ANOMALY_MODEL_REPROCESS'
    WHEN 'ANOMALY_AUTODISCOVERY' THEN 'ANOMALY_MODEL_REFRESH'
    ELSE job_type
END
WHERE job_type IN ('BUILD_MODEL', 'FIND_ANOMALIES', 'ANOMALY_AUTODISCOVERY');

UPDATE trendz_task_sequence_item
SET reference_type = CASE reference_type
    WHEN 'ANOMALY_MODEL_BUILD_MODEL' THEN 'ANOMALY_MODEL_BUILD'
    WHEN 'ANOMALY_MODEL_FIND_ANOMALIES' THEN 'ANOMALY_MODEL_REPROCESS'
    WHEN 'ANOMALY_MODEL_ANOMALY_AUTODISCOVERY' THEN 'ANOMALY_MODEL_REFRESH'
    ELSE reference_type
END
WHERE reference_type IN (
    'ANOMALY_MODEL_BUILD_MODEL',
    'ANOMALY_MODEL_FIND_ANOMALIES',
    'ANOMALY_MODEL_ANOMALY_AUTODISCOVERY'
);

DO $$
DECLARE
    column_exists BOOLEAN;
BEGIN
    SELECT EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_name = 'cluster_model' AND column_name = 'autodiscover_enabled'
    ) INTO column_exists;

    IF column_exists THEN
        EXECUTE $sql$
        INSERT INTO anomaly_model_task_data (
            anomaly_model_id,
            enabled_refresh,
            enabled_save_to_tb,
            enabled_alarm_creation,
            item_set_json,
            refresh_time_unit,
            refresh_time_unit_count,
            refresh_time_unit_truncated
        )
        SELECT id, autodiscover_enabled, FALSE, FALSE, NULL, unit, unit_count, FALSE
        FROM cluster_model
        WHERE autodiscover_enabled = TRUE
          AND unit_count > 0
          AND unit IS NOT NULL
          AND NOT EXISTS (
              SELECT 1 FROM anomaly_model_task_data WHERE anomaly_model_id = cluster_model.id
          )
        $sql$;
    END IF;
END $$;

ALTER TABLE cluster_model
    DROP COLUMN IF EXISTS autodiscover_enabled,
    DROP COLUMN IF EXISTS autodiscover_job_id,
    DROP COLUMN IF EXISTS unit,
    DROP COLUMN IF EXISTS unit_count;
ALTER TABLE dataset_config
    DROP COLUMN IF EXISTS business_entity_type;

DROP INDEX IF EXISTS cluster_model_autodiscover_job_id_idx;
DROP INDEX IF EXISTS runtime_filter_field_dataset_config_id_idx;

CREATE TABLE IF NOT EXISTS api_key (
    id          UUID NOT NULL CONSTRAINT api_key_pkey PRIMARY KEY,
    tenant_id   UUID NOT NULL,
    token       VARCHAR(255) NOT NULL,

    UNIQUE (tenant_id), UNIQUE (token)
);

CREATE INDEX IF NOT EXISTS api_key_tenant_id_idx    ON api_key (tenant_id);
CREATE INDEX IF NOT EXISTS api_key_token_idx        ON api_key (token);

UPDATE trendz_task
SET schedule_scheduling_time_zone = 'Europe/Kyiv'
WHERE schedule_scheduling_time_zone = 'Europe/Kiev';

WITH best_unit AS (
  SELECT
    cm.id,
    CASE
      WHEN (
        (mp.json_value::jsonb -> 'segmentSplitProperties' ->> 'type') = 'FIXED_TIME'
      )
      THEN (mp.json_value::jsonb -> 'segmentSplitProperties' ->> 'windowSizeMs')::BIGINT
      ELSE (
        ((mp.json_value::jsonb -> 'segmentSplitProperties' ->> 'windowSizeMs')::BIGINT *
         (mp.json_value::jsonb -> 'segmentSplitProperties' ->> 'slidingStepPercent')::FLOAT / 100)::BIGINT
      )
    END AS max_ts
  FROM cluster_model cm
  JOIN ml_properties mp ON mp.id = cm.properties_id
  WHERE cm.telemetry_save_period_unit IS NULL
)
UPDATE cluster_model cm
SET telemetry_save_period_unit = CASE
  WHEN bu.max_ts >= 86400000 THEN 'day'
  ELSE 'hour'
END
FROM best_unit bu
WHERE cm.id = bu.id;

UPDATE cluster_model
SET alarm_type = FORMAT('Anomaly from ''%s'' anomaly model', name)
WHERE alarm_type IS NULL;

UPDATE cluster_model
SET tb_telemetry_key = LEFT(
    FORMAT(
        'telemetry_key_for_%s_anomaly_model',
        REPLACE(name, ' ', '_')
    ), 64
)
WHERE tb_telemetry_key IS NULL;

WITH min_scores AS (
  SELECT
    model_id,
    MIN(score) AS min_score
  FROM anomaly
  WHERE score IS NOT NULL
  GROUP BY model_id
)
UPDATE cluster_model cm
SET
  alarm_warning_threshold    = FLOOR(ms.min_score * 1.2),
  alarm_minor_threshold      = FLOOR(ms.min_score * 1.6),
  alarm_major_threshold      = FLOOR(ms.min_score * 3.0),
  alarm_critical_threshold   = FLOOR(ms.min_score * 4.0)
FROM min_scores ms
WHERE cm.id = ms.model_id;

UPDATE view_assistance_model_config
SET json_properties = jsonb_build_object(
    'apiKey', (json_properties::jsonb)->'customProperties'->>'apiKey',
    'baseUrl', (json_properties::jsonb)->'customProperties'->>'baseUrl',
    'modelName', (json_properties::jsonb)->'customProperties'->>'modelName'
) WHERE model_provider = 'CUSTOM' AND (json_properties::jsonb) ? 'customProperties';

UPDATE view_assistance_model_config
SET json_properties = jsonb_build_object(
    'apiKey', (json_properties::jsonb)->'openAiProperties'->>'apiKey'
) WHERE model_provider = 'OPEN_AI' AND (json_properties::jsonb) ? 'openAiProperties';

UPDATE view_assistance_model_config
SET json_properties = jsonb_build_object(
    'apiKey', (json_properties::jsonb)->'googleProperties'->>'apiKey'
) WHERE model_provider = 'GOOGLE' AND (json_properties::jsonb) ? 'googleProperties';

UPDATE view_assistance_model_config
SET json_properties = jsonb_build_object(
    'region', (json_properties::jsonb)->'awsProperties'->>'region',
    'accessKey', (json_properties::jsonb)->'awsProperties'->>'accessKey',
    'secretKey', (json_properties::jsonb)->'awsProperties'->>'secretKey'
) WHERE model_provider = 'AMAZON_BEDROCK' AND (json_properties::jsonb) ? 'awsProperties';

DO $$
BEGIN
    IF EXISTS (
        SELECT 1
        FROM information_schema.table_constraints
        WHERE constraint_name = 'runtime_filter_field_dataset_config_id_fkey'
          AND table_name = 'runtime_filter_field'
          AND constraint_type = 'FOREIGN KEY'
    ) THEN
        ALTER TABLE runtime_filter_field
        DROP CONSTRAINT runtime_filter_field_dataset_config_id_fkey;
    END IF;
END $$;


ALTER TABLE calculation_field ADD COLUMN IF NOT EXISTS split_time_range BOOLEAN NOT NULL DEFAULT false;
ALTER TABLE calculation_field ADD COLUMN IF NOT EXISTS split_time_range_time_unit VARCHAR(32) NOT NULL DEFAULT 'DAY';
