/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.trendz.subscription;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Expiry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.thingsboard.server.common.data.id.TenantId;
import org.thingsboard.trendz.domain.customize.UserRecord;
import org.thingsboard.trendz.security.entity.JwtSecurityUser;
import org.thingsboard.trendz.security.service.AuthenticationService;
import org.thingsboard.trendz.service.customize.UserManagementService;
import org.thingsboard.trendz.service.provider.TbRestDataSource;
import org.thingsboard.trendz.subscription.SubscriptionService;
import org.thingsboard.trendz.subscription.SubscriptionState;
import org.thingsboard.trendz.subscription.SubscriptionType;
import org.thingsboard.trendz.tools.DonReactive;
import reactor.core.publisher.Mono;

public class CloudLicenseChecker
implements SubscriptionService {
    private static final Logger log = LoggerFactory.getLogger(CloudLicenseChecker.class);
    private final UserManagementService userManagementService;
    private final AuthenticationService authenticationService;
    private final TbRestDataSource tbRestDataSource;
    private final Cache<TenantId, SubscriptionState> cache;

    public CloudLicenseChecker(UserManagementService userManagementService, AuthenticationService authenticationService, TbRestDataSource tbRestDataSource) {
        this.userManagementService = userManagementService;
        this.authenticationService = authenticationService;
        this.tbRestDataSource = tbRestDataSource;
        this.cache = this.buildCache();
    }

    public SubscriptionState getSubscriptionState(JwtSecurityUser securityUser) {
        TenantId tenantId = securityUser.getTenant();
        return (SubscriptionState)this.cache.get((Object)tenantId, id -> this.checkIsValid(securityUser));
    }

    public void stop() {
    }

    private SubscriptionState checkIsValid(JwtSecurityUser securityUser) {
        long now = System.currentTimeMillis();
        try {
            log.info("Going to check license for tenant {} / {}", (Object)securityUser.getTenantId(), (Object)securityUser.getUserId());
            JwtSecurityUser user = securityUser;
            if (securityUser.isCustomerUser()) {
                log.trace("Check license for customer user");
                UserRecord userRecord = (UserRecord)this.userManagementService.findFirstTenantUserByTenantId(securityUser.getTenantId()).orElseThrow(() -> new RuntimeException("Tenant not found in Trendz cache"));
                user = userRecord.getUser();
            }
            String jwtToken = this.authenticationService.getToken(user);
            JsonNode subscription = this.fetchSubscription(jwtToken);
            JsonNode details = this.fetchSubscriptionDetails(jwtToken);
            JsonNode filteredSubscription = this.makeFilteredCopy(subscription);
            this.userManagementService.assignLicenceInfo(user, filteredSubscription);
            String name = subscription.get("name").asText();
            SubscriptionType type = this.makeSubscriptionType(name);
            boolean licenseActive = subscription.get("active").booleanValue();
            boolean trendzEnabled = details.get("configuration").get("trendzEnabled").booleanValue();
            int maxAssets = details.get("configuration").get("maxAssets").intValue();
            int maxDevices = details.get("configuration").get("maxDevices").intValue();
            boolean isValid = licenseActive && trendzEnabled;
            log.info("Tenant {} trendzEnabled {} licenseActive {}", new Object[]{user.getTenantId(), trendzEnabled, licenseActive});
            String errorMessage = null;
            if (licenseActive && !trendzEnabled) {
                errorMessage = "The current license does not include Trendz. Current subscription: " + name;
            }
            return SubscriptionState.builder().valid(isValid).checkTs(now).actualAssets(-1L).allowedAssets((long)maxAssets).actualDevices(-1L).allowedDevices((long)maxDevices).type(type).name(name).errorMsg(errorMessage).build();
        }
        catch (Exception ex) {
            String errorMessage = String.format("License check error for %s: %s", securityUser.getTenantId(), ex.getMessage());
            log.error(errorMessage, (Throwable)ex);
            return SubscriptionState.builder().valid(false).checkTs(now).errorMsg(errorMessage).build();
        }
    }

    private JsonNode fetchSubscriptionDetails(String jwtToken) {
        try {
            return (JsonNode)DonReactive.block((Mono)this.tbRestDataSource.getSubscriptionDetails(jwtToken));
        }
        catch (RuntimeException ex) {
            throw new RuntimeException("Could not fetch subscription details", ex);
        }
    }

    private JsonNode fetchSubscription(String jwtToken) {
        try {
            return (JsonNode)DonReactive.block((Mono)this.tbRestDataSource.getSubscription(jwtToken));
        }
        catch (RuntimeException ex) {
            throw new RuntimeException("Could not fetch subscription", ex);
        }
    }

    private Cache<TenantId, SubscriptionState> buildCache() {
        return Caffeine.newBuilder().expireAfter((Expiry)new /* Unavailable Anonymous Inner Class!! */).build();
    }

    private JsonNode makeFilteredCopy(JsonNode subscription) {
        ObjectNode copy = (ObjectNode)subscription.deepCopy();
        copy.remove("id");
        copy.remove("additionalInfo");
        copy.remove("externalId");
        copy.remove("tenantId");
        copy.remove("billingCustomerId");
        copy.remove("subscriptionPlanId");
        return copy;
    }

    private SubscriptionType makeSubscriptionType(String name) {
        if (name.contains("Trendz")) {
            if (name.contains("Maker")) {
                return SubscriptionType.MAKER;
            }
            if (name.contains("Weekly")) {
                return SubscriptionType.WEEKLY;
            }
            if (name.contains("Prototype")) {
                return SubscriptionType.PROTOTYPE;
            }
            if (name.contains("Startup")) {
                return SubscriptionType.STARTUP;
            }
            if (name.contains("Business")) {
                return SubscriptionType.BUSINESS;
            }
            if (name.contains("Business+")) {
                return SubscriptionType.BUSINESS_PLUS;
            }
            if (name.contains("Enterprise")) {
                return SubscriptionType.ENTERPRISE;
            }
        }
        return SubscriptionType.UNKNOWN;
    }
}

