/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.trendz.subscription;

import java.text.SimpleDateFormat;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.thingsboard.license.client.TbLicenseClient;
import org.thingsboard.license.client.TbLicenseClientListener;
import org.thingsboard.license.shared.exception.LicenseErrorCode;
import org.thingsboard.license.shared.exception.LicenseException;
import org.thingsboard.trendz.security.entity.JwtSecurityUser;
import org.thingsboard.trendz.security.service.AuthenticationService;
import org.thingsboard.trendz.service.customize.UserManagementService;
import org.thingsboard.trendz.service.provider.TbAssetService;
import org.thingsboard.trendz.service.provider.TbDeviceService;
import org.thingsboard.trendz.subscription.DiscoverySubscriptionUtils;
import org.thingsboard.trendz.subscription.SubscriptionPlanData;
import org.thingsboard.trendz.subscription.SubscriptionService;
import org.thingsboard.trendz.subscription.SubscriptionState;
import org.thingsboard.trendz.subscription.SubscriptionType;

public class BaseSubscriptionService
implements SubscriptionService,
TbLicenseClientListener {
    private static final Logger log = LoggerFactory.getLogger(BaseSubscriptionService.class);
    private static final String MAX_DEVICES_KEY = "device_count";
    private static final String MAX_ASSETS_KEY = "asset_count";
    private static final long CHECK_INTERVAL = TimeUnit.HOURS.toMillis(12L);
    private final String licenseSecret;
    private final String instanceDataFilePath;
    private final TbAssetService assetService;
    private final TbDeviceService deviceService;
    private final AuthenticationService authenticationService;
    private final UserManagementService userManagementService;
    private final ExecutorService triggerCheckScheduler;
    private final ConcurrentMap<UUID, SubscriptionState> tenantIdSubscriptionStateMap = new ConcurrentHashMap();
    private TbLicenseClient tbLicenseClient;

    public BaseSubscriptionService(String licenseSecret, String instanceDataFilePath, TbAssetService assetService, TbDeviceService deviceService, AuthenticationService authenticationService, UserManagementService userManagementService) {
        this.licenseSecret = licenseSecret;
        this.instanceDataFilePath = instanceDataFilePath;
        this.assetService = assetService;
        this.deviceService = deviceService;
        this.authenticationService = authenticationService;
        this.userManagementService = userManagementService;
        this.triggerCheckScheduler = Executors.newSingleThreadExecutor();
        this.init();
    }

    public void stop() {
        if (this.tbLicenseClient != null) {
            this.tbLicenseClient.stop();
        }
        this.triggerCheckScheduler.shutdownNow();
    }

    public SubscriptionState getSubscriptionState(JwtSecurityUser securityUser) {
        SubscriptionState state = (SubscriptionState)this.tenantIdSubscriptionStateMap.get(securityUser.getTenantId());
        if (state == null || !state.isValid() || System.currentTimeMillis() - state.getCheckTs() > CHECK_INTERVAL) {
            this.triggerCheckScheduler.submit(() -> {
                SubscriptionState currentState = (SubscriptionState)this.tenantIdSubscriptionStateMap.get(securityUser.getTenantId());
                if (currentState == null || !currentState.isValid() || System.currentTimeMillis() - currentState.getCheckTs() > CHECK_INTERVAL) {
                    try {
                        log.debug("Validate subscription. Current state {}", (Object)currentState);
                        if (this.tbLicenseClient != null) {
                            SubscriptionState newState = this.checkSubscription(securityUser);
                            this.tenantIdSubscriptionStateMap.put(securityUser.getTenantId(), newState);
                            log.info("Subscription validated {}", (Object)newState);
                        } else {
                            this.tenantIdSubscriptionStateMap.put(securityUser.getTenantId(), this.makeInvalidSubscriptionState("License client is not initialized"));
                            log.error("License client is not initialized");
                        }
                    }
                    catch (RuntimeException re) {
                        this.tenantIdSubscriptionStateMap.put(securityUser.getTenantId(), this.makeInvalidSubscriptionState("Could not validate subscription"));
                        log.error("Could not validate subscription", (Throwable)re);
                    }
                }
            });
        }
        return state == null ? SubscriptionState.builder().valid(true).build() : state;
    }

    public void onError(LicenseException e) {
        String errorMessage = String.format("License Error occurred: %s(%s) - %s", e.getErrorCode(), e.getErrorCode().getErrorCode(), e.getMessage());
        log.error(errorMessage, (Throwable)e);
        if (e.isCritical()) {
            this.tenantIdSubscriptionStateMap.clear();
        }
    }

    private void init() {
        if (StringUtils.isBlank((CharSequence)this.licenseSecret)) {
            String errorMessage = "License secret is not provided! Please provide license.secret property value in trendz.yml or set TRENDZ_LICENSE_SECRET environment variable!";
            log.error(errorMessage);
        } else {
            try {
                this.tbLicenseClient = TbLicenseClient.builder().listener((TbLicenseClientListener)this).licenseSecret(this.licenseSecret).licenseDataFilePath(this.instanceDataFilePath).releaseDate(new SimpleDateFormat("yyyy-MM-dd").parse("2025-11-05").getTime()).build();
                this.tbLicenseClient.init();
            }
            catch (Exception e) {
                LicenseErrorCode licenseErrorCode = e instanceof LicenseException ? ((LicenseException)e).getErrorCode() : LicenseErrorCode.GENERAL_ERROR;
                String errorMessage = String.format("Failed to init license client (code = %s): %s", licenseErrorCode, e.getMessage());
                log.error(errorMessage, (Throwable)e);
                this.tbLicenseClient = null;
            }
        }
    }

    private SubscriptionState checkSubscription(JwtSecurityUser user) {
        Page usersPage;
        UUID tenantId = user.getTenantId();
        int page = 0;
        int pageSize = 10;
        do {
            usersPage = this.userManagementService.findAllTenantUserByTenantId(tenantId, page, pageSize);
            for (JwtSecurityUser tenantUser : usersPage.getContent()) {
                try {
                    return this.checkSubscriptionByTenantUser(tenantUser);
                }
                catch (Exception e) {
                    log.error("Can not validate subscription by given user: {}", (Object)tenantUser, (Object)e);
                }
            }
            ++page;
        } while (usersPage.hasNext());
        throw new RuntimeException("Can not validate subscription by all users of tenant: " + String.valueOf(tenantId));
    }

    private SubscriptionState checkSubscriptionByTenantUser(JwtSecurityUser tenantUser) {
        long now = System.currentTimeMillis();
        String jwtToken = this.authenticationService.getToken(tenantUser);
        long countAssets = this.countAssets(jwtToken);
        long countDevices = this.countDevices(jwtToken);
        long maxAssetsCount = this.tbLicenseClient.getPlanLongValue(MAX_ASSETS_KEY);
        long maxDevicesCount = this.tbLicenseClient.getPlanLongValue(MAX_DEVICES_KEY);
        SubscriptionPlanData planData = new SubscriptionPlanData(maxAssetsCount, maxDevicesCount);
        SubscriptionType subscriptionType = DiscoverySubscriptionUtils.getSubscriptionType((SubscriptionPlanData)planData).orElse(SubscriptionType.UNKNOWN);
        String name = "Self-hosted subscription: %s".formatted(subscriptionType.getValue());
        if (maxAssetsCount > 0L && maxAssetsCount < countAssets) {
            String msg = "Maximum allowed assets limit reached! Current %s but allowed %s".formatted(countAssets, maxAssetsCount);
            log.error(msg);
            return SubscriptionState.builder().valid(false).checkTs(now).actualAssets(countAssets).actualDevices(countDevices).allowedAssets(maxAssetsCount).allowedDevices(maxDevicesCount).type(subscriptionType).name(name).errorMsg(msg).build();
        }
        if (maxDevicesCount > 0L && maxDevicesCount < countDevices) {
            String msg = "Maximum allowed devices limit reached! Current %s but allowed %s".formatted(countDevices, maxDevicesCount);
            log.error(msg);
            return SubscriptionState.builder().valid(false).checkTs(now).actualAssets(countAssets).actualDevices(countDevices).allowedAssets(maxAssetsCount).allowedDevices(maxDevicesCount).type(subscriptionType).name(name).errorMsg(msg).build();
        }
        return SubscriptionState.builder().valid(true).checkTs(now).actualAssets(countAssets).actualDevices(countDevices).allowedAssets(maxAssetsCount).allowedDevices(maxDevicesCount).type(subscriptionType).name(name).errorMsg(null).build();
    }

    private long countDevices(String jwtToken) {
        long count = 0L;
        Set deviceTypes = this.deviceService.loadDeviceTypes(jwtToken);
        for (String deviceType : deviceTypes) {
            count += (long)this.deviceService.loadDeviceByType(deviceType, jwtToken).size();
        }
        return count;
    }

    private long countAssets(String jwtToken) {
        long count = 0L;
        Set assetTypes = this.assetService.loadAssetTypes(jwtToken);
        for (String assetType : assetTypes) {
            count += (long)this.assetService.loadAssetByType(assetType, jwtToken).size();
        }
        return count;
    }

    private SubscriptionState makeInvalidSubscriptionState(String errorMessage) {
        long now = System.currentTimeMillis();
        return SubscriptionState.builder().valid(false).checkTs(now).errorMsg(errorMessage).build();
    }
}

