/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.trendz.service.task;

import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.thingsboard.trendz.dao.TimeStampUUIDGenerator;
import org.thingsboard.trendz.dao.task.TaskDao;
import org.thingsboard.trendz.dao.task.TaskExecutionState;
import org.thingsboard.trendz.dao.task.TaskState;
import org.thingsboard.trendz.domain.definition.view.config.DateAggregationUnit;
import org.thingsboard.trendz.domain.definition.view.config.ViewConfigTaskStatusData;
import org.thingsboard.trendz.exception.TrendzInternalException;
import org.thingsboard.trendz.exception.task.TaskExecutionNotFoundException;
import org.thingsboard.trendz.exception.task.TaskExecutionNotStartedYetException;
import org.thingsboard.trendz.exception.task.TaskExecutionNullIdException;
import org.thingsboard.trendz.exception.task.TaskNotFoundException;
import org.thingsboard.trendz.exception.task.TaskNullIdException;
import org.thingsboard.trendz.exception.task.TrendzTaskException;
import org.thingsboard.trendz.security.entity.JwtSecurityUser;
import org.thingsboard.trendz.service.task.TaskJob;
import org.thingsboard.trendz.service.task.TaskService;
import org.thingsboard.trendz.service.task.model.Task;
import org.thingsboard.trendz.service.task.model.TaskConfig;
import org.thingsboard.trendz.service.task.model.TaskExecution;
import org.thingsboard.trendz.service.task.model.TaskExecutionFilteringField;
import org.thingsboard.trendz.service.task.model.TaskExecutionLite;
import org.thingsboard.trendz.service.task.model.TaskExecutionRequest;
import org.thingsboard.trendz.service.task.model.TaskExecutionSortingField;
import org.thingsboard.trendz.service.task.model.TaskExecutionStatus;
import org.thingsboard.trendz.service.task.model.TaskFilteringField;
import org.thingsboard.trendz.service.task.model.TaskJobType;
import org.thingsboard.trendz.service.task.model.TaskLite;
import org.thingsboard.trendz.service.task.model.TaskReference;
import org.thingsboard.trendz.service.task.model.TaskRetryPolicy;
import org.thingsboard.trendz.service.task.model.TaskSchedule;
import org.thingsboard.trendz.service.task.model.TaskScheduleType;
import org.thingsboard.trendz.service.task.model.TaskSortingField;
import org.thingsboard.trendz.service.task.model.TaskStoreExecutionConfig;
import org.thingsboard.trendz.service.task.model.TaskTimeoutConfig;
import org.thingsboard.trendz.service.task.model.TaskTtlConfig;

@Service
public class TaskService {
    private static final Logger log = LoggerFactory.getLogger(TaskService.class);
    private final TaskDao taskDao;
    private final boolean cloud;

    @Autowired
    public TaskService(TaskDao taskDao, @Value(value="${deployment.cloud}") boolean cloud) {
        this.taskDao = taskDao;
        this.cloud = cloud;
    }

    public Page<TaskLite> getAllTasksLite(JwtSecurityUser user, int page, int pageSize, List<TaskFilteringField> filteringFields, List<TaskSortingField> sortingFields) {
        return this.taskDao.getAllLite(user, page, pageSize, filteringFields, sortingFields);
    }

    public Page<Task> getAllTasks(JwtSecurityUser user, int page, int pageSize, List<TaskFilteringField> filteringFields, List<TaskSortingField> sortingFields) {
        return this.taskDao.getAll(user, page, pageSize, filteringFields, sortingFields);
    }

    public Optional<Task> findTaskById(JwtSecurityUser user, UUID taskId) {
        return this.findTaskById(user, taskId, false);
    }

    public Optional<Task> findTaskById(JwtSecurityUser user, UUID taskId, boolean insecure) {
        if (taskId == null) {
            throw new TaskNullIdException();
        }
        return this.taskDao.findById(user, taskId, insecure);
    }

    public Page<TaskExecutionLite> getAllTaskExecutionsLite(JwtSecurityUser user, int page, int pageSize, List<TaskExecutionFilteringField> filteringFields, List<TaskExecutionSortingField> sortingFields) {
        return this.taskDao.getAllExecutionsLite(user, page, pageSize, filteringFields, sortingFields);
    }

    public Page<TaskExecutionLite> getAllTaskExecutionsByTaskLite(JwtSecurityUser user, UUID taskId, int page, int pageSize, List<TaskExecutionFilteringField> filteringFields, List<TaskExecutionSortingField> sortingFields) {
        if (taskId == null) {
            throw new TaskNullIdException();
        }
        return this.taskDao.getAllExecutionsByTaskLite(user, taskId, page, pageSize, filteringFields, sortingFields);
    }

    public Optional<TaskExecution> findExecutionById(JwtSecurityUser user, UUID executionId) {
        if (executionId == null) {
            throw new TaskExecutionNullIdException();
        }
        return this.taskDao.findExecutionById(user, executionId);
    }

    public Optional<Task> findTaskByReferencedEntity(JwtSecurityUser user, TaskReference reference) {
        return this.findTaskByReferencedEntity(user, reference, false);
    }

    public Optional<Task> findTaskByReferencedEntity(JwtSecurityUser user, TaskReference reference, boolean insecure) {
        return this.taskDao.findTaskByReferencedEntity(user, reference, insecure);
    }

    public TaskExecution pollExecutionById(JwtSecurityUser user, UUID executionId) {
        return this.findExecutionById(user, executionId).orElseGet(() -> {
            this.taskDao.findExecutionRequest(user, executionId).ifPresentOrElse(record -> {
                throw new TaskExecutionNotStartedYetException(executionId);
            }, () -> {
                throw new TaskExecutionNotFoundException(executionId);
            });
            return null;
        });
    }

    public Task createTask(JwtSecurityUser user, TaskConfig taskConfig) {
        Task task = this.makeTaskFromConfig(user, taskConfig);
        return this.rawSave(task);
    }

    public Task updateTask(JwtSecurityUser user, TaskConfig taskConfig) {
        TaskJobType newType;
        UUID taskId = taskConfig.getId();
        Task task = (Task)this.findTaskById(user, taskId).orElseThrow(() -> new TaskNotFoundException(taskId));
        TaskJobType currentType = task.getJobType();
        if (currentType != (newType = taskConfig.getJob().getJobType())) {
            throw new TrendzTaskException("You can not change task job type");
        }
        this.modifyTaskByConfig(user, task, taskConfig);
        this.validateSchedule(task);
        return this.taskDao.saveTask(task);
    }

    public void deleteTask(JwtSecurityUser user, UUID taskId) {
        if (taskId == null) {
            throw new TaskNullIdException();
        }
        Task task = (Task)this.findTaskById(user, taskId).orElseThrow(() -> new TaskNotFoundException(taskId));
        this.cancelExecutionsByTask(user, task, true);
        this.taskDao.deleteTask(user, taskId);
        this.taskDao.removeTaskScheduleRecord(taskId);
    }

    public Task rawSave(Task task) {
        this.taskDao.saveTaskScheduleRecord(task.getId(), TaskState.FREE, 0L);
        this.validateSchedule(task);
        Task saved = this.taskDao.saveTask(task);
        log.debug("A new task was created: {}", (Object)saved.getName());
        return saved;
    }

    public UUID runExecution(JwtSecurityUser user, UUID taskId, boolean cancelOthers) {
        return this.runExecution(user, taskId, null, cancelOthers, false);
    }

    public UUID runExecution(JwtSecurityUser user, UUID taskId, TaskJob job, boolean cancelOthers) {
        return this.runExecution(user, taskId, job, cancelOthers, false);
    }

    public UUID runExecution(JwtSecurityUser user, UUID taskId, TaskJob job, boolean cancelOthers, boolean insecure) {
        if (taskId == null) {
            throw new TaskNullIdException();
        }
        Task task = (Task)this.findTaskById(user, taskId, insecure).orElseThrow(() -> new TaskNotFoundException(taskId));
        if (cancelOthers) {
            this.cancelExecutionsByTask(user, task, false);
        }
        UUID newExecutionId = TimeStampUUIDGenerator.generateId();
        long now = System.currentTimeMillis();
        TaskJob usedJob = job == null ? task.getJob() : job;
        TaskExecutionRequest request = TaskExecutionRequest.builder().taskId(taskId).executionId(newExecutionId).scheduled(false).user(user).job(usedJob).createdTs(now).state(TaskState.FREE).build();
        this.taskDao.saveExecutionRequest(request);
        return newExecutionId;
    }

    public void cancelExecutionById(JwtSecurityUser user, UUID executionId) {
        if (executionId == null) {
            throw new TaskExecutionNullIdException();
        }
        TaskExecution execution = (TaskExecution)this.findExecutionById(user, executionId).orElseThrow(() -> new TaskExecutionNotFoundException(executionId));
        this.taskDao.saveTaskExecutionStateRecords(execution.getId(), TaskExecutionState.CANCELLED, false);
    }

    public void cancelExecutionsByTaskId(JwtSecurityUser user, UUID taskId) {
        if (taskId == null) {
            throw new TaskNullIdException();
        }
        Task task = (Task)this.findTaskById(user, taskId).orElseThrow(() -> new TaskNotFoundException(taskId));
        this.cancelExecutionsByTask(user, task, false);
    }

    public void cancelAllExecutions(JwtSecurityUser user) {
        this.taskDao.cancelTaskExecutionsForUser(user);
    }

    public TaskExecution awaitExecution(JwtSecurityUser user, UUID executionId) throws TimeoutException {
        return this.awaitExecution(user, executionId, TimeUnit.HOURS.toMillis(1L), TimeUnit.SECONDS.toMillis(3L));
    }

    public TaskExecution awaitExecution(JwtSecurityUser user, UUID executionId, long timeoutMs, long intervalMs) throws TimeoutException {
        long startTs = System.currentTimeMillis();
        do {
            try {
                log.debug("Waiting for execution {} to finish, time to timeout = {}", (Object)executionId, (Object)(timeoutMs - (System.currentTimeMillis() - startTs)));
                TimeUnit.MILLISECONDS.sleep(intervalMs);
                TaskExecution taskExecution = this.pollExecutionById(user, executionId);
                TaskExecutionStatus status = taskExecution.getStatus();
                if (!status.isCancellable()) {
                    return taskExecution;
                }
            }
            catch (TaskExecutionNotStartedYetException taskExecution) {
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        } while (System.currentTimeMillis() - startTs < timeoutMs);
        throw new TimeoutException();
    }

    public void deleteAllExecutions(JwtSecurityUser user) {
        this.taskDao.deleteAllExecutions(user);
    }

    public Optional<Long> getTaskSchedulingTime(JwtSecurityUser user, UUID taskId) {
        Task task = (Task)this.findTaskById(user, taskId, false).orElseThrow(() -> new TaskNotFoundException(taskId));
        return this.taskDao.getTaskSchedulingTime(task.getId());
    }

    public Optional<Long> getTaskSchedulingTime(JwtSecurityUser user, TaskReference reference) {
        Task task = (Task)this.findTaskByReferencedEntity(user, reference, false).orElseThrow(() -> new TaskNotFoundException(reference));
        return this.taskDao.getTaskSchedulingTime(task.getId());
    }

    public void resetTaskScheduling(JwtSecurityUser user, UUID taskId) {
        Task task = (Task)this.findTaskById(user, taskId, false).orElseThrow(() -> new TaskNotFoundException(taskId));
        this.cancelExecutionsByTaskId(user, task.getId());
        this.taskDao.saveTaskScheduleRecord(task.getId(), TaskState.FREE, 0L);
    }

    public List<ViewConfigTaskStatusData> getViewConfigStatusData(UUID tenantId, Set<UUID> configIdSet) {
        return this.taskDao.getViewConfigStatusData(tenantId, configIdSet);
    }

    public TaskSchedule makeSchedulingConfig(int unitCount, DateAggregationUnit timeUnit, boolean isTruncated, String tzName) {
        if (timeUnit == null) {
            log.warn("Undefined task schedule, use 1 day period.");
            return TaskSchedule.builder().type(TaskScheduleType.PERIODIC).periodMs(TimeUnit.DAYS.toMillis(1L)).schedulingUnit(ChronoUnit.HOURS).schedulingUnitCount(unitCount).schedulingTimeZone(tzName).build();
        }
        if (isTruncated) {
            return TaskSchedule.builder().type(TaskScheduleType.SCHEDULED).schedulingUnit(timeUnit.mapToChronoUnit()).schedulingUnitCount(unitCount).schedulingTimeZone(tzName).build();
        }
        TaskSchedule.TaskScheduleBuilder builder = TaskSchedule.builder().type(TaskScheduleType.PERIODIC).schedulingUnit(ChronoUnit.HOURS).schedulingUnitCount(unitCount).schedulingTimeZone(tzName);
        return switch (1.$SwitchMap$org$thingsboard$trendz$domain$definition$view$config$DateAggregationUnit[timeUnit.ordinal()]) {
            case 1 -> builder.periodMs(TimeUnit.DAYS.toMillis(30L * (long)unitCount)).build();
            case 2 -> builder.periodMs(TimeUnit.DAYS.toMillis(7L * (long)unitCount)).build();
            case 3 -> builder.periodMs(TimeUnit.DAYS.toMillis(unitCount)).build();
            case 4 -> builder.periodMs(TimeUnit.HOURS.toMillis(unitCount)).build();
            case 5 -> {
                if (this.cloud) {
                    throw new TrendzInternalException("Unit not supported : " + String.valueOf(timeUnit));
                }
                yield builder.periodMs(TimeUnit.MINUTES.toMillis(unitCount)).build();
            }
            default -> throw new TrendzInternalException("Unit not supported : " + String.valueOf(timeUnit));
        };
    }

    private void cancelExecutionsByTask(JwtSecurityUser user, Task task, boolean removedTask) {
        this.taskDao.cancelTaskExecutionsForTask(user, task.getId(), removedTask);
    }

    private String generateName(Task task) {
        return String.format("Task: type = %s, reference = %s, schedule = %s", task.getJob().getJobType(), task.getReference().getType(), task.getSchedule().getType());
    }

    private Task makeTaskFromConfig(JwtSecurityUser user, TaskConfig taskConfig) {
        UUID taskId = TimeStampUUIDGenerator.generateId();
        long now = System.currentTimeMillis();
        Task task = Task.builder().id(taskId).user(user).createdTs(now).updatedTs(now).enabled(Objects.requireNonNullElse(taskConfig.getEnabled(), true).booleanValue()).schedule(Objects.requireNonNullElse(taskConfig.getSchedule(), TaskSchedule.getDefault())).ttlConfig(Objects.requireNonNullElse(taskConfig.getTtlConfig(), TaskTtlConfig.getDefault())).storeExecutionConfig(Objects.requireNonNullElse(taskConfig.getStoreExecutionConfig(), TaskStoreExecutionConfig.getDefault())).timeoutConfig(Objects.requireNonNullElse(taskConfig.getTimeoutConfig(), TaskTimeoutConfig.getDefault())).retryPolicy(Objects.requireNonNullElse(taskConfig.getRetryPolicy(), TaskRetryPolicy.getDefault())).jobType(taskConfig.getJob().getJobType()).job(taskConfig.getJob()).reference(Objects.requireNonNullElse(taskConfig.getReference(), TaskReference.getDefault())).executions(new ArrayList()).build();
        task.setName(Objects.requireNonNullElse(taskConfig.getName(), this.generateName(task)));
        return task;
    }

    private void modifyTaskByConfig(JwtSecurityUser user, Task task, TaskConfig taskConfig) {
        TaskConfig newConfig = TaskConfig.builder().enabled(Objects.requireNonNullElse(taskConfig.getEnabled(), true)).schedule(Objects.requireNonNullElse(taskConfig.getSchedule(), TaskSchedule.getDefault())).ttlConfig(Objects.requireNonNullElse(taskConfig.getTtlConfig(), TaskTtlConfig.getDefault())).timeoutConfig(Objects.requireNonNullElse(taskConfig.getTimeoutConfig(), TaskTimeoutConfig.getDefault())).retryPolicy(Objects.requireNonNullElse(taskConfig.getRetryPolicy(), TaskRetryPolicy.getDefault())).job(taskConfig.getJob()).reference(Objects.requireNonNullElse(taskConfig.getReference(), TaskReference.getDefault())).build();
        task.setEnabled(newConfig.getEnabled().booleanValue());
        task.setSchedule(newConfig.getSchedule());
        task.setTtlConfig(newConfig.getTtlConfig());
        task.setTimeoutConfig(newConfig.getTimeoutConfig());
        task.setRetryPolicy(newConfig.getRetryPolicy());
        task.setJob(newConfig.getJob());
        task.setReference(newConfig.getReference());
        task.setName(Objects.requireNonNullElse(taskConfig.getName(), this.generateName(task)));
        long now = System.currentTimeMillis();
        task.setUpdatedTs(now);
        task.setUser(user);
    }

    public void validateSchedule(Task task) {
        TaskSchedule schedule = task.getSchedule();
        if (schedule.getSchedulingUnit() == null || schedule.getSchedulingUnit().compareTo(ChronoUnit.HOURS) < 0 && this.cloud || schedule.getSchedulingUnit().compareTo(ChronoUnit.MINUTES) < 0) {
            throw new IllegalArgumentException("Invalid schedule unit: " + String.valueOf(schedule.getSchedulingUnit()));
        }
        if (schedule.getSchedulingUnitCount() < 1) {
            throw new IllegalArgumentException("Invalid schedule unit count (must be > 1): " + schedule.getSchedulingUnitCount());
        }
    }
}

