/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.trendz.service.task;

import com.google.common.collect.Sets;
import java.time.Duration;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.reactivestreams.Subscription;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import org.thingsboard.trendz.dao.TimeStampUUIDGenerator;
import org.thingsboard.trendz.dao.task.TaskDao;
import org.thingsboard.trendz.dao.task.TaskExecutionState;
import org.thingsboard.trendz.dao.task.TaskExecutionStateRecordDto;
import org.thingsboard.trendz.dao.task.TaskState;
import org.thingsboard.trendz.domain.customize.UserRecord;
import org.thingsboard.trendz.exception.task.TaskExecutionEmptyResultException;
import org.thingsboard.trendz.exception.task.TaskExecutionException;
import org.thingsboard.trendz.exception.task.TaskExecutionWrongReturnTypeException;
import org.thingsboard.trendz.exception.task.TaskNotFoundException;
import org.thingsboard.trendz.security.entity.JwtSecurityUser;
import org.thingsboard.trendz.service.customize.UserManagementService;
import org.thingsboard.trendz.service.executor.ExecutorManagementService;
import org.thingsboard.trendz.service.executor.ExecutorName;
import org.thingsboard.trendz.service.metrics.TaskMetricService;
import org.thingsboard.trendz.service.startup.TrendzStartupService;
import org.thingsboard.trendz.service.task.TaskExecutionProgressContent;
import org.thingsboard.trendz.service.task.TaskExecutionProgressService;
import org.thingsboard.trendz.service.task.TaskExecutionProgressStepBuilder;
import org.thingsboard.trendz.service.task.TaskJob;
import org.thingsboard.trendz.service.task.TaskJobExecutor;
import org.thingsboard.trendz.service.task.TaskJobExecutorManager;
import org.thingsboard.trendz.service.task.model.Task;
import org.thingsboard.trendz.service.task.model.TaskExecution;
import org.thingsboard.trendz.service.task.model.TaskExecutionProgress;
import org.thingsboard.trendz.service.task.model.TaskExecutionRequest;
import org.thingsboard.trendz.service.task.model.TaskExecutionStatus;
import org.thingsboard.trendz.service.task.model.TaskJobType;
import org.thingsboard.trendz.subscription.SubscriptionService;
import org.thingsboard.trendz.tools.json.JsonUtils;
import reactor.core.Disposable;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Scheduler;

@Service
public class TaskRuntimeService
implements TrendzStartupService {
    private static final Logger log = LoggerFactory.getLogger(TaskRuntimeService.class);
    private final TaskDao taskDao;
    private final TaskJobExecutorManager jobExecutorManager;
    private final TaskExecutionProgressService taskExecutionProgressService;
    private final TaskMetricService metricService;
    private final UserManagementService userManagementService;
    private final SubscriptionService subscriptionService;
    private final boolean enable;
    private final boolean enableBackground;
    private final boolean ttlTrackingEnable;
    private final boolean lostTrackingEnable;
    private final int taskQuerySize;
    private final int defaultTaskTimeout;
    private final int stateRecordTimeout;
    private final ScheduledExecutorService schedulingExecutor;
    private final Map<UUID, TaskExecution> executions;
    private final Map<UUID, Subscription> taskExecutionIdToSubscriptionMap;
    private final Set<UUID> executionsToRemove;

    @Autowired
    public TaskRuntimeService(ExecutorManagementService executorManagementService, TaskDao taskDao, TaskJobExecutorManager jobExecutorManager, TaskExecutionProgressService taskExecutionProgressService, TaskMetricService metricService, UserManagementService userManagementService, SubscriptionService subscriptionService, @Value(value="${taskService.enable}") boolean enable, @Value(value="${taskService.enableBackground}") boolean enableBackground, @Value(value="${taskService.ttlTrackingEnable}") boolean ttlTrackingEnable, @Value(value="${taskService.lostTrackingEnable}") boolean lostTrackingEnable, @Value(value="${taskService.taskQuerySize}") int taskQuerySize, @Value(value="${taskService.defaultTaskTimeout}") int defaultTaskTimeout, @Value(value="${taskService.stateRecordTimeout}") int stateRecordTimeout) {
        this.taskDao = taskDao;
        this.jobExecutorManager = jobExecutorManager;
        this.taskExecutionProgressService = taskExecutionProgressService;
        this.metricService = metricService;
        this.userManagementService = userManagementService;
        this.subscriptionService = subscriptionService;
        this.enable = enable;
        this.enableBackground = enableBackground;
        this.ttlTrackingEnable = ttlTrackingEnable;
        this.lostTrackingEnable = lostTrackingEnable;
        this.taskQuerySize = taskQuerySize;
        this.defaultTaskTimeout = defaultTaskTimeout;
        this.stateRecordTimeout = stateRecordTimeout;
        this.schedulingExecutor = (ScheduledExecutorService)executorManagementService.getExecutorByName(ExecutorName.TASK_SERVICE_SCHEDULING);
        this.executions = new ConcurrentHashMap();
        this.taskExecutionIdToSubscriptionMap = new ConcurrentHashMap();
        this.executionsToRemove = ConcurrentHashMap.newKeySet();
    }

    public void onStartup(ApplicationContext context) {
        if (this.enable) {
            log.info("The task runtime scheduler is enabled!");
            this.schedulingExecutor.scheduleWithFixedDelay(() -> this.scheduleLogs(), 0L, 10L, TimeUnit.SECONDS);
            this.schedulingExecutor.scheduleWithFixedDelay(() -> this.scheduleCancellation(), 0L, 100L, TimeUnit.MILLISECONDS);
            this.schedulingExecutor.scheduleWithFixedDelay(() -> this.scheduleRefresh(), 0L, 1L, TimeUnit.SECONDS);
            if (this.enableBackground) {
                this.schedulingExecutor.scheduleWithFixedDelay(() -> this.scheduleDelayed(), 0L, 1L, TimeUnit.SECONDS);
            } else {
                log.warn("The background task scheduler is disabled!");
            }
            this.schedulingExecutor.scheduleWithFixedDelay(() -> this.scheduleProcess(), 0L, 100L, TimeUnit.MILLISECONDS);
            if (this.ttlTrackingEnable) {
                this.schedulingExecutor.scheduleWithFixedDelay(() -> this.scheduleTtl(), 0L, 10L, TimeUnit.SECONDS);
            } else {
                log.warn("The task TTL scheduler is disabled!");
            }
            if (this.lostTrackingEnable) {
                this.schedulingExecutor.scheduleWithFixedDelay(() -> this.scheduleLost(), 0L, 5L, TimeUnit.SECONDS);
            } else {
                log.warn("The lost task scheduler is disabled!");
            }
        } else {
            log.warn("The task runtime scheduler is disabled!");
        }
    }

    public int priority() {
        return 40;
    }

    private void scheduleLogs() {
        try {
            log.debug("Task runtime cycle: current executions = {}/{}", (Object)this.executions.size(), (Object)this.taskQuerySize);
        }
        catch (Exception e) {
            log.error("Error during task scheduling: logs.", (Throwable)e);
        }
    }

    private void scheduleTtl() {
        try {
            this.taskDao.removeExpiredTaskExecutionsByTtl();
        }
        catch (Exception e) {
            log.error("Error during task scheduling: TTL.", (Throwable)e);
        }
    }

    private void scheduleLost() {
        try {
            this.taskDao.processLostTaskExecutions((long)this.stateRecordTimeout);
        }
        catch (Exception e) {
            log.error("Error during task scheduling: Lost.", (Throwable)e);
        }
    }

    private void scheduleCancellation() {
        try {
            HashSet currentExecutionIds = new HashSet(this.taskExecutionIdToSubscriptionMap.keySet());
            Set cancelledTaskExecutionIds = this.taskDao.findCancelledTaskExecutionIds(currentExecutionIds);
            if (!cancelledTaskExecutionIds.isEmpty()) {
                for (TaskExecutionStateRecordDto stateRecordDto : cancelledTaskExecutionIds) {
                    this.cancelTaskExecution(stateRecordDto.getExecutionId());
                    if (!stateRecordDto.isRemovedTask()) continue;
                    this.executionsToRemove.add(stateRecordDto.getExecutionId());
                }
                log.info("Task cancel: {} executions were cancelled", (Object)cancelledTaskExecutionIds.size());
            }
        }
        catch (Exception e) {
            log.error("Error during task scheduling: Cancellation.", (Throwable)e);
        }
    }

    private void scheduleRefresh() {
        try {
            HashSet currentExecutionIds = new HashSet(this.taskExecutionIdToSubscriptionMap.keySet());
            Set cancelledTaskExecutionIds = this.taskDao.findCancelledTaskExecutionIds(currentExecutionIds).stream().map(TaskExecutionStateRecordDto::getExecutionId).collect(Collectors.toSet());
            Sets.SetView workingExecutionIds = Sets.difference(currentExecutionIds, cancelledTaskExecutionIds);
            this.taskDao.updateTaskExecutionStateRecords((Set)workingExecutionIds, TaskExecutionState.WORKING);
        }
        catch (Exception e) {
            log.error("Error during task scheduling: Refresh.", (Throwable)e);
        }
    }

    private void scheduleDelayed() {
        try {
            this.taskDao.findAndBorrowTaskAndDoAction(arg_0 -> this.createExecutionRequest(arg_0));
        }
        catch (Exception e) {
            log.error("Error during task scheduling: Delayed.", (Throwable)e);
        }
    }

    private void scheduleProcess() {
        try {
            if (this.executions.size() < this.taskQuerySize) {
                Optional executionRequest = this.taskDao.findAndBorrowTaskExecutionRequest();
                executionRequest.ifPresent(arg_0 -> this.startTaskExecution(arg_0));
            }
        }
        catch (Exception e) {
            log.error("Error during task scheduling: Process.", (Throwable)e);
        }
    }

    private void createExecutionRequest(Task task) {
        long now = System.currentTimeMillis();
        TaskExecutionRequest request = TaskExecutionRequest.builder().taskId(task.getId()).executionId(TimeStampUUIDGenerator.generateId()).scheduled(true).user(task.getUser()).job(task.getJob()).createdTs(now).state(TaskState.FREE).build();
        this.taskDao.saveExecutionRequest(request);
    }

    private void startTaskExecution(TaskExecutionRequest request) {
        boolean valid;
        long now = System.currentTimeMillis();
        UUID taskId = request.getTaskId();
        UUID executionId = request.getExecutionId();
        Task task = (Task)this.taskDao.findById(request.getUser(), taskId, true).orElseThrow(() -> new TaskNotFoundException(taskId));
        JwtSecurityUser user = request.getUser();
        TaskJob oldJob = request.getJob();
        TaskJob newJob = oldJob.clone();
        TaskJobType jobType = newJob.getJobType();
        TaskJobExecutor jobExecutor = this.jobExecutorManager.getExecutorByType(jobType);
        Scheduler scheduler = jobExecutor.getExecuteScheduler();
        Consumer<String> skipTaskExecution = errMsg -> this.skipTaskExecution(request, executionId, taskId, user, now, jobType, oldJob, task, errMsg);
        if (request.isScheduled() && !(valid = this.userManagementService.validateUser(user))) {
            Optional recordOpt = this.userManagementService.findRecordByUser(user);
            String userName = recordOpt.map(UserRecord::getUsername).orElse("UNKNOWN USER");
            skipTaskExecution.accept("The user is invalid, user = " + userName);
            return;
        }
        if (!this.subscriptionService.getSubscriptionState(user).isValid()) {
            log.warn("Trendz License not valid. Skip task execution for tenant {}", (Object)user.getTenant());
            skipTaskExecution.accept("Trendz License not valid");
            return;
        }
        TaskExecutionProgressContent progressContent = jobExecutor.createProgressContent();
        TaskExecutionProgressStepBuilder rootStepBuilder = TaskExecutionProgressStepBuilder.createRootStepBuilder((TaskExecutionProgressService)this.taskExecutionProgressService, (UUID)taskId, (UUID)executionId, (TaskExecutionProgressContent)progressContent);
        TaskExecution execution = TaskExecution.builder().id(executionId).taskId(taskId).user(user).status(TaskExecutionStatus.CREATED).createdTs(now).taskExecutionProgress(new TaskExecutionProgress(progressContent, null)).jobType(jobType).job(oldJob).jsonResult("").build();
        this.taskDao.saveTaskExecutionStateRecords(executionId, TaskExecutionState.WORKING, false);
        this.taskDao.saveExecution(execution);
        this.taskDao.removeExecutionRequest(taskId, executionId);
        Disposable disposable = Mono.just((Object)executionId).doOnNext(o -> log.trace("Execution chain: created")).publishOn(scheduler).doOnNext(o -> log.trace("Execution chain: published")).doOnNext(o2 -> this.callbackStart(user, task, execution)).doOnNext(o -> log.trace("Execution chain: started")).flatMap(o2 -> jobExecutor.execute(user, newJob, rootStepBuilder)).doOnNext(o -> log.trace("Execution chain: pre-finish")).flatMap(arg_0 -> this.checkResultType(arg_0)).map(JsonUtils::toJson).timeout(this.getTaskDuration(task)).doOnNext(o -> log.trace("Execution chain: finish")).switchIfEmpty(Mono.error((Throwable)new TaskExecutionEmptyResultException())).doOnCancel(() -> {
            jobExecutor.doOnCancel(user, newJob, rootStepBuilder);
            this.callbackCancel(user, task, execution);
            this.callbackComplete(user, request, task, execution, rootStepBuilder, true);
            this.subscriptionRemove(task, execution);
        }).doOnError(throwable -> jobExecutor.doOnError(user, newJob, rootStepBuilder)).onErrorMap(TaskExecutionException::new).subscribe(result -> this.callbackFinish(user, task, execution, result, oldJob, newJob), throwable -> {
            this.callbackFailed(user, task, execution, throwable);
            this.callbackComplete(user, request, task, execution, rootStepBuilder, true);
            this.subscriptionRemove(task, execution);
        }, () -> {
            this.callbackComplete(user, request, task, execution, rootStepBuilder, true);
            this.subscriptionRemove(task, execution);
        }, subscription -> {
            this.subscriptionRegister(execution, subscription);
            subscription.request(1L);
        });
    }

    private void skipTaskExecution(TaskExecutionRequest request, UUID executionId, UUID taskId, JwtSecurityUser user, long now, TaskJobType jobType, TaskJob oldJob, Task task, String resultMsg) {
        TaskExecution execution = TaskExecution.builder().id(executionId).taskId(taskId).user(user).status(TaskExecutionStatus.CANCELED).createdTs(now).startTs(now).finishTs(now).taskExecutionProgress(new TaskExecutionProgress()).jobType(jobType).job(oldJob).jsonResult(new TaskExecutionException((Throwable)new IllegalStateException(resultMsg)).toString()).build();
        this.taskDao.updateTaskEnabled(taskId, false);
        this.taskDao.removeExecutionRequest(taskId, executionId);
        this.callbackCancel(user, task, execution);
        this.callbackComplete(user, request, task, execution, null, false);
    }

    private void cancelTaskExecution(UUID executionId) {
        Subscription subscription = (Subscription)this.taskExecutionIdToSubscriptionMap.get(executionId);
        subscription.cancel();
    }

    private Duration getTaskDuration(Task task) {
        if (task.getTimeoutConfig().isEnabled()) {
            long taskDuration = task.getTimeoutConfig().getDuration();
            long minDuration = Math.min((long)this.defaultTaskTimeout, taskDuration);
            return Duration.ofMillis(minDuration);
        }
        return Duration.ofMillis(this.defaultTaskTimeout);
    }

    private <T> Mono<T> checkResultType(T result) {
        Class<?> returnType = result.getClass();
        if (Mono.class.isAssignableFrom(returnType) || Flux.class.isAssignableFrom(returnType)) {
            return Mono.error((Throwable)new TaskExecutionWrongReturnTypeException());
        }
        return Mono.just(result);
    }

    private void subscriptionRegister(TaskExecution execution, Subscription subscription) {
        this.executions.put(execution.getId(), execution);
        this.taskExecutionIdToSubscriptionMap.put(execution.getId(), subscription);
        log.debug("The task execution subscription is registered, execution id = {}, task id = {}", (Object)execution.getId(), (Object)execution.getTaskId());
    }

    private void subscriptionRemove(Task task, TaskExecution execution) {
        this.executions.remove(execution.getId());
        this.taskExecutionIdToSubscriptionMap.remove(execution.getId());
        this.metricService.removeExecution(task, execution);
        log.debug("The task execution subscription is removed, execution id = {}, task id = {}", (Object)execution.getId(), (Object)execution.getTaskId());
    }

    private void callbackStart(JwtSecurityUser user, Task task, TaskExecution execution) {
        execution.setStartTs(System.currentTimeMillis());
        execution.setStatus(TaskExecutionStatus.RUNNING);
        this.taskDao.saveExecution(execution);
        this.metricService.registerExecution(task, execution);
        log.debug("The task execution was STARTED, task id = {}, task name = {}", (Object)task.getId(), (Object)task.getName());
    }

    private void callbackFinish(JwtSecurityUser user, Task task, TaskExecution execution, String jsonResult, TaskJob oldJob, TaskJob newJob) {
        execution.setStatus(TaskExecutionStatus.FINISHED);
        execution.setJsonResult(jsonResult);
        if (!oldJob.equals(newJob)) {
            log.debug("The task job was changed.");
            this.taskDao.updateTaskJob(task.getId(), newJob);
        }
        log.debug("The task execution was FINISHED, task id = {}, task name = {}", (Object)task.getId(), (Object)task.getName());
    }

    private void callbackFailed(JwtSecurityUser user, Task task, TaskExecution execution, Throwable throwable) {
        execution.setStatus(TaskExecutionStatus.FAILED);
        if (throwable instanceof TaskExecutionException) {
            TaskExecutionException exception = (TaskExecutionException)throwable;
            if (exception.getCause() instanceof TimeoutException) {
                String message = String.format("The task execution was interrupted by timeout: %s ms", task.getTimeoutConfig().getDuration());
                execution.setJsonResult(message);
                log.error("The task execution was FAILED, task id = {}, task name = {}", new Object[]{task.getId(), task.getName(), exception});
            } else {
                execution.setJsonResult(exception.toString());
                log.error("The task execution was FAILED, task id = {}, task name = {}", new Object[]{task.getId(), task.getName(), exception});
            }
        } else {
            String message = String.format("%s: %s", throwable.getClass().getName(), throwable.getMessage());
            execution.setJsonResult(message);
            log.error("The task execution - CRITICAL ERROR, task id = {}, task name = {}", new Object[]{task.getId(), task.getName(), throwable});
        }
    }

    private void callbackCancel(JwtSecurityUser user, Task task, TaskExecution execution) {
        execution.setStatus(TaskExecutionStatus.CANCELED);
        log.warn("The task execution was CANCELLED, task id = {}, task name = {}", (Object)task.getId(), (Object)task.getName());
    }

    private void callbackComplete(JwtSecurityUser user, TaskExecutionRequest request, Task task, TaskExecution execution, TaskExecutionProgressStepBuilder rootStepBuilder, boolean removeExecutionStateRecord) {
        if (rootStepBuilder != null) {
            rootStepBuilder.close();
        }
        long now = System.currentTimeMillis();
        UUID executionId = execution.getId();
        if (removeExecutionStateRecord) {
            this.taskDao.removeTaskExecutionStateRecord(executionId);
        }
        if (this.executionsToRemove.contains(executionId)) {
            this.executionsToRemove.remove(executionId);
            return;
        }
        execution.setFinishTs(now);
        execution.setDuration(execution.getStartTs() == 0L ? 0L : execution.getFinishTs() - execution.getStartTs());
        this.taskDao.saveExecution(execution);
        if (request.isScheduled()) {
            this.taskDao.saveTaskScheduleRecord(task.getId(), TaskState.FREE, now);
        }
    }
}

