/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.trendz.service.graph;

import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.stream.Streams;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.thingsboard.trendz.domain.definition.entity.BusinessEntity;
import org.thingsboard.trendz.domain.definition.entity.relation.Direction;
import org.thingsboard.trendz.domain.definition.entity.relation.Relation;
import org.thingsboard.trendz.exception.TrendzException;
import org.thingsboard.trendz.service.graph.RelationEdge;
import org.thingsboard.trendz.service.graph.RelationGraph;
import org.thingsboard.trendz.service.graph.RelationGraphService;
import org.thingsboard.trendz.service.graph.RelationNode;

@Service
public class RelationGraphService {
    private static final Logger log = LoggerFactory.getLogger(RelationGraphService.class);

    public RelationGraph createGraph(List<BusinessEntity> entities) {
        HashSet<RelationNode> nodes = new HashSet<RelationNode>();
        HashSet<RelationEdge> edges = new HashSet<RelationEdge>();
        for (BusinessEntity entity : entities) {
            List relations = entity.getRelations();
            RelationNode node = RelationNode.builder().entityId(entity.getId()).entityName(entity.getName()).build();
            nodes.add(node);
            for (Relation relation : relations) {
                if (relation.getDirection() != Direction.FROM) continue;
                RelationEdge edge = RelationEdge.builder().edgeId(UUID.randomUUID()).entityFrom(entity.getId()).entityTo(relation.getRelatedEntityId()).type(relation.getRelationType()).enabled(relation.isEnabled()).build();
                edges.add(edge);
            }
        }
        return new RelationGraph(nodes, edges);
    }

    public RelationGraph reduceGraph(RelationGraph graph, Set<UUID> allowedEntitySet) {
        Set allNodeIds = this.getAllNodeIds(graph);
        Sets.SetView difference = Sets.difference(allowedEntitySet, (Set)allNodeIds);
        if (!difference.isEmpty()) {
            String message = String.format("You trying to limit set of entities by unknown entities: %s", Arrays.toString(difference.toArray()));
            throw new TrendzException(message);
        }
        Set filteredNodes = graph.getNodes().stream().filter(node -> allowedEntitySet.contains(node.getEntityId())).map(RelationNode::new).collect(Collectors.toSet());
        Set filteredEdges = graph.getEdges().stream().filter(edge -> allowedEntitySet.contains(edge.getEntityFrom()) && allowedEntitySet.contains(edge.getEntityTo())).map(RelationEdge::new).collect(Collectors.toSet());
        return new RelationGraph(filteredNodes, filteredEdges);
    }

    public RelationGraph getMinimalGraph(RelationGraph graph, Set<UUID> requiredEntitySet) {
        if (requiredEntitySet.isEmpty()) {
            return new RelationGraph(new HashSet(), new HashSet());
        }
        Set allNodeIds = this.getAllNodeIds(graph);
        Sets.SetView difference = Sets.difference(requiredEntitySet, (Set)allNodeIds);
        if (!difference.isEmpty()) {
            String message = String.format("You trying to make minimal graph by foreign entities: %s", Arrays.toString(difference.toArray()));
            throw new TrendzException(message);
        }
        Map adjacencyMap = this.createAdjacencyMap(graph);
        Map invertedAdjacencyMap = this.createInvertedAdjacencyMap(graph);
        HashSet allNodes = new HashSet();
        HashSet allEdges = new HashSet();
        for (UUID startNode : requiredEntitySet) {
            HashMap<UUID, Set<List<UUID>>> pathMap = new HashMap<UUID, Set<List<UUID>>>();
            LinkedList<MinimalTreeSearchItem> deque = new LinkedList<MinimalTreeSearchItem>();
            Supplier<MinimalTreeSearchItem> pollFromDeque = deque::pollFirst;
            Set<UUID> startVisited = Set.of(startNode);
            List<UUID> startPath = List.of(startNode);
            deque.addLast(new MinimalTreeSearchItem(startNode, startVisited, startPath));
            pathMap.put(startNode, Set.of(List.of(startNode)));
            while (!deque.isEmpty()) {
                MinimalTreeSearchItem currentItem = pollFromDeque.get();
                UUID currentNode = currentItem.getNodeId();
                Set currentVisited = currentItem.getVisited();
                List currentPath = currentItem.getPath();
                int currentDistance = currentPath.size();
                Set inNeighbors = adjacencyMap.getOrDefault(currentNode, Collections.emptySet());
                Set outNeighbors = invertedAdjacencyMap.getOrDefault(currentNode, Collections.emptySet());
                Sets.SetView neighbors = Sets.union(inNeighbors, outNeighbors);
                for (UUID neighbor : neighbors) {
                    int newRequiredVisitedNodeCount;
                    int newDistance;
                    Set pathSet;
                    if (currentVisited.contains(neighbor)) continue;
                    Sets.SetView newVisited = Sets.union((Set)currentVisited, Set.of(neighbor));
                    List newPath = Stream.concat(currentPath.stream(), Stream.of(neighbor)).collect(Collectors.toList());
                    MinimalTreeSearchItem searchItem = new MinimalTreeSearchItem(neighbor, (Set)newVisited, newPath);
                    if (!pathMap.containsKey(neighbor)) {
                        pathSet = new HashSet();
                        pathSet.add(newPath);
                        pathMap.put(neighbor, pathSet);
                        deque.addLast(searchItem);
                        continue;
                    }
                    pathSet = (Set)pathMap.get(neighbor);
                    List path2 = (List)pathSet.iterator().next();
                    int prevRequiredVisitedNodeCount = Sets.intersection(requiredEntitySet, new HashSet(path2)).size();
                    int prevDistance = path2.size() - prevRequiredVisitedNodeCount;
                    if (prevDistance == (newDistance = currentDistance + 1 - (newRequiredVisitedNodeCount = Sets.intersection(requiredEntitySet, (Set)newVisited).size()))) {
                        pathSet.add(newPath);
                        deque.addLast(searchItem);
                        continue;
                    }
                    if (prevDistance <= newDistance) continue;
                    HashSet newPathSet = new HashSet();
                    newPathSet.add(newPath);
                    pathMap.put(neighbor, newPathSet);
                    deque.addLast(searchItem);
                }
            }
            Map<UUID, Set> finalPathMap = pathMap.entrySet().stream().filter(entry -> requiredEntitySet.contains(entry.getKey())).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
            Set visitedNodes = finalPathMap.values().stream().flatMap(Collection::stream).flatMap(Collection::stream).collect(Collectors.toSet());
            Set visitedEdges = finalPathMap.values().stream().flatMap(Collection::stream).map(path -> {
                HashSet<Pair> pairSet = new HashSet<Pair>();
                for (int i = 0; i < path.size() - 1; ++i) {
                    UUID from = (UUID)path.get(i);
                    UUID to = (UUID)path.get(i + 1);
                    pairSet.add(Pair.of((Object)from, (Object)to));
                }
                return pairSet;
            }).flatMap(Collection::stream).collect(Collectors.toSet());
            allNodes.addAll(visitedNodes);
            allEdges.addAll(visitedEdges);
        }
        Map edgeMap = this.createEgdeMap(graph);
        Set nodes = graph.getNodes().stream().filter(node -> allNodes.contains(node.getEntityId())).collect(Collectors.toSet());
        Set edges = allEdges.stream().map(pair -> edgeMap.getOrDefault(pair.getLeft(), Collections.emptyMap()).getOrDefault(pair.getRight(), Collections.emptySet())).flatMap(Collection::stream).collect(Collectors.toSet());
        return new RelationGraph(nodes, edges);
    }

    public RelationGraph getFullConnectedComponentFromRootEntities(RelationGraph fullRelationGraph, Set<UUID> rootEntities) {
        Map sourceNodeMap = fullRelationGraph.getNodes().stream().collect(Collectors.toMap(RelationNode::getEntityId, Function.identity()));
        Set rootNodes = rootEntities.stream().filter(sourceNodeMap::containsKey).map(sourceNodeMap::get).collect(Collectors.toSet());
        Map sourceEdgeMap = Streams.of((Object[])new Stream[]{fullRelationGraph.getEdges().stream().map(relationEdge -> Pair.of((Object)relationEdge.getEntityFrom(), (Object)relationEdge.getEntityTo())), fullRelationGraph.getEdges().stream().map(relationEdge -> Pair.of((Object)relationEdge.getEntityTo(), (Object)relationEdge.getEntityFrom()))}).flatMap(Function.identity()).collect(Collectors.groupingBy(Pair::getKey, Collectors.mapping(Pair::getValue, Collectors.toSet())));
        Set newNodes = new HashSet(rootNodes);
        while (!newNodes.isEmpty()) {
            newNodes = newNodes.stream().map(RelationNode::getEntityId).map(sourceEdgeMap::get).filter(Objects::nonNull).flatMap(Collection::stream).map(sourceNodeMap::get).collect(Collectors.toSet());
            newNodes.removeAll(rootNodes);
            rootNodes.addAll(newNodes);
        }
        Map sourceRootNodeMap = rootNodes.stream().collect(Collectors.toMap(RelationNode::getEntityId, Function.identity()));
        Set rootEdges = fullRelationGraph.getEdges().stream().filter(relationEdge -> sourceRootNodeMap.containsKey(relationEdge.getEntityFrom())).collect(Collectors.toSet());
        return new RelationGraph(rootNodes, rootEdges);
    }

    public Set<RelationGraph> findAllConnectedComponents(RelationGraph relationGraph) {
        HashSet<RelationGraph> connectedComponents = new HashSet<RelationGraph>();
        RelationGraph graph = relationGraph;
        while (!graph.getNodes().isEmpty()) {
            RelationNode anyNode = (RelationNode)graph.getNodes().stream().findAny().orElseThrow();
            RelationGraph connectedComponent = this.getFullConnectedComponentFromRootEntities(graph, Set.of(anyNode.getEntityId()));
            connectedComponents.add(connectedComponent);
            Sets.SetView newNodes = Sets.difference((Set)graph.getNodes(), (Set)connectedComponent.getNodes());
            Sets.SetView newEdges = Sets.difference((Set)graph.getEdges(), (Set)connectedComponent.getEdges());
            graph = new RelationGraph((Set)newNodes, (Set)newEdges);
        }
        return connectedComponents;
    }

    public Set<UUID> getEntitySet(RelationGraph graph) {
        return this.getAllNodeIds(graph);
    }

    public UUID findCentralNode(RelationGraph graph, Set<UUID> entityIds) {
        if (entityIds.isEmpty()) {
            throw new TrendzException("Node Id set is empty");
        }
        Set allNodeIds = this.getAllNodeIds(graph);
        Sets.SetView difference = Sets.difference(entityIds, (Set)allNodeIds);
        if (!difference.isEmpty()) {
            String message = String.format("You trying to find central node of graph by unknown entities: %s", Arrays.toString(difference.toArray()));
            throw new TrendzException(message);
        }
        Map entityFromToEntityToMap = this.createAdjacencyMap(graph);
        Map entityToToEntityFromMap = this.createInvertedAdjacencyMap(graph);
        HashMap<UUID, Integer> centralityScores = new HashMap<UUID, Integer>();
        Set allNodes = allNodeIds.stream().filter(entityIds::contains).collect(Collectors.toSet());
        for (UUID nodeId : allNodes) {
            GraphSearchResult bfsResult = this.bfs(entityFromToEntityToMap, entityToToEntityFromMap, nodeId);
            Map distances = this.calculateDistancesFromPath(bfsResult);
            int totalDistance = distances.values().stream().mapToInt(Integer::intValue).sum();
            centralityScores.put(nodeId, totalDistance);
        }
        Map.Entry minEntry = Collections.min(centralityScores.entrySet(), Map.Entry.comparingByValue());
        return (UUID)minEntry.getKey();
    }

    public Map<UUID, Integer> computeDistancesFromRoot(RelationGraph graph, UUID rootEntityId) {
        Map entityFromToEntityToMap = this.createAdjacencyMap(graph);
        Map entityToToEntityFromMap = this.createInvertedAdjacencyMap(graph);
        GraphSearchResult bfsResult = this.bfs(entityFromToEntityToMap, entityToToEntityFromMap, rootEntityId);
        Map distances = this.calculateDistancesFromPath(bfsResult);
        return distances;
    }

    public Optional<UUID> getParentEntity(RelationGraph graph, UUID rootId, UUID entityId) {
        if (rootId.equals(entityId)) {
            return Optional.empty();
        }
        Map adjacencyMap = this.createAdjacencyMap(graph);
        Map invertedAdjacencyMap = this.createInvertedAdjacencyMap(graph);
        GraphSearchResult bfsResult = this.bfs(adjacencyMap, invertedAdjacencyMap, rootId);
        Map paths = bfsResult.getPaths();
        List pathFromRootToEntity = (List)paths.get(entityId);
        int pathLength = pathFromRootToEntity.size();
        UUID parentId = (UUID)pathFromRootToEntity.get(pathLength - 2);
        return Optional.of(parentId);
    }

    public void validateGraph(RelationGraph graph) {
        this.validateConsistency(graph);
        this.validateSingleConnectedComponent(graph);
        this.validateAmbiguousRelation(graph);
        this.validateCycleAbsence(graph);
    }

    private void validateConsistency(RelationGraph graph) {
        Set nodeIds = this.getAllNodeIds(graph);
        for (RelationEdge edge : graph.getEdges()) {
            if (nodeIds.contains(edge.getEntityFrom()) && nodeIds.contains(edge.getEntityTo())) continue;
            throw new TrendzException("Graph is not consistent: Edge points to non-existent node!");
        }
        if (nodeIds.size() != graph.getNodes().size()) {
            throw new TrendzException("Graph is not consistent: Duplicate node IDs found!");
        }
        Set edgeIds = graph.getEdges().stream().map(RelationEdge::getEdgeId).collect(Collectors.toSet());
        if (edgeIds.size() != graph.getEdges().size()) {
            throw new TrendzException("Graph is not consistent: Duplicate edge IDs found!");
        }
    }

    private void validateSingleConnectedComponent(RelationGraph graph) {
        RelationNode node;
        Map entityToToEntityFromMap;
        Set nodes = graph.getNodes();
        if (nodes.isEmpty()) {
            return;
        }
        Map entityFromToEntityToMap = this.createAdjacencyMap(graph);
        GraphSearchResult dfsResult = this.dfs(entityFromToEntityToMap, entityToToEntityFromMap = this.createInvertedAdjacencyMap(graph), (node = (RelationNode)nodes.iterator().next()).getEntityId());
        Set visited = dfsResult.getVisited();
        if (visited.size() != nodes.size()) {
            String message = String.format("Graph is not valid: the graph is not single connected component. Node count = %s, component size = %s", nodes.size(), visited.size());
            throw new TrendzException(message);
        }
    }

    private void validateAmbiguousRelation(RelationGraph graph) {
        Map edgeMap = this.createEgdeMap(graph);
        List invalidEdges = edgeMap.values().stream().map(Map::values).flatMap(Collection::stream).flatMap(Collection::stream).map(relationEdge -> new GraphEdge(relationEdge.getEntityFrom(), relationEdge.getEntityTo())).collect(Collectors.groupingBy(Function.identity(), Collectors.counting())).entrySet().stream().filter(entry -> (Long)entry.getValue() != 1L).map(Map.Entry::getKey).collect(Collectors.toList());
        if (!invalidEdges.isEmpty()) {
            Map<UUID, String> idToNameMap = graph.getNodes().stream().collect(Collectors.toMap(RelationNode::getEntityId, RelationNode::getEntityName));
            Set edgesValues = invalidEdges.stream().map(GraphEdge::getPair).map(pair -> String.format("(%s - %s)", idToNameMap.get(pair.getLeft()), idToNameMap.get(pair.getRight()))).collect(Collectors.toSet());
            String message = String.format("Graph is not valid: ambiguous relations were found: %s", Arrays.toString(edgesValues.toArray()));
            throw new TrendzException(message);
        }
    }

    private void validateCycleAbsence(RelationGraph graph) {
        Map adjacencyMap = this.createAdjacencyMap(graph);
        Map invertedAdjacencyMap = this.createInvertedAdjacencyMap(graph);
        HashSet<UUID> generalVisited = new HashSet<UUID>();
        HashMap<UUID, UUID> idToParentId = new HashMap<UUID, UUID>();
        LinkedList<UUID> deque = new LinkedList<UUID>();
        for (RelationNode node : graph.getNodes()) {
            UUID entityId = node.getEntityId();
            if (generalVisited.contains(entityId)) continue;
            HashSet<UUID> currentVisited = new HashSet<UUID>();
            deque.addLast(entityId);
            while (!deque.isEmpty()) {
                UUID currentNode = (UUID)deque.pollLast();
                if (!generalVisited.contains(currentNode)) {
                    generalVisited.add(currentNode);
                    currentVisited.add(currentNode);
                    Set inNeighbors = adjacencyMap.getOrDefault(currentNode, Collections.emptySet());
                    Set outNeighbors = invertedAdjacencyMap.getOrDefault(currentNode, Collections.emptySet());
                    LinkedHashSet neighbors = new LinkedHashSet(inNeighbors);
                    neighbors.addAll(outNeighbors);
                    neighbors.remove(idToParentId.get(currentNode));
                    for (UUID neighbor : neighbors) {
                        if (currentVisited.contains(neighbor)) {
                            Map<UUID, String> idToNameMap = graph.getNodes().stream().collect(Collectors.toMap(RelationNode::getEntityId, RelationNode::getEntityName));
                            ArrayList<UUID> cycle = new ArrayList<UUID>();
                            UUID cycleIterator = currentNode;
                            while (!cycleIterator.equals(neighbor)) {
                                cycle.add(cycleIterator);
                                cycleIterator = (UUID)idToParentId.get(cycleIterator);
                            }
                            cycle.add(cycleIterator);
                            List cycleString = cycle.stream().map(idToNameMap::get).collect(Collectors.toList());
                            String message = String.format("Graph is not valid: cycle was found for the next entities: %s", Arrays.toString(cycleString.toArray()));
                            throw new TrendzException(message);
                        }
                        if (generalVisited.contains(neighbor)) continue;
                        deque.addLast(neighbor);
                        idToParentId.put(neighbor, currentNode);
                    }
                    continue;
                }
                currentVisited.remove(currentNode);
            }
        }
    }

    private Set<UUID> getAllNodeIds(RelationGraph graph) {
        return graph.getNodes().stream().map(RelationNode::getEntityId).collect(Collectors.toSet());
    }

    private Map<UUID, Set<UUID>> createAdjacencyMap(RelationGraph graph) {
        return graph.getEdges().stream().filter(RelationEdge::isEnabled).collect(Collectors.groupingBy(RelationEdge::getEntityFrom, Collectors.mapping(RelationEdge::getEntityTo, Collectors.toSet())));
    }

    private Map<UUID, Set<UUID>> createInvertedAdjacencyMap(RelationGraph graph) {
        return graph.getEdges().stream().filter(RelationEdge::isEnabled).collect(Collectors.groupingBy(RelationEdge::getEntityTo, Collectors.mapping(RelationEdge::getEntityFrom, Collectors.toSet())));
    }

    private Map<UUID, Map<UUID, Set<RelationEdge>>> createEgdeMap(RelationGraph graph) {
        return graph.getEdges().stream().filter(RelationEdge::isEnabled).collect(Collectors.groupingBy(RelationEdge::getEntityFrom, Collectors.groupingBy(RelationEdge::getEntityTo, Collectors.toSet())));
    }

    private Map<UUID, Integer> calculateDistancesFromPath(GraphSearchResult bfsResult) {
        Map paths = bfsResult.getPaths();
        return paths.entrySet().stream().map(entry -> {
            UUID entityId = (UUID)entry.getKey();
            int distance = ((List)entry.getValue()).size();
            return Map.entry(entityId, distance);
        }).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    private GraphSearchResult graphSearch(Map<UUID, Set<UUID>> adjacencyMap, Map<UUID, Set<UUID>> invertedAdjacencyMap, UUID startNode, Deque<UUID> deque, Supplier<UUID> pollFromDeque) {
        HashMap<UUID, List<UUID>> paths = new HashMap<UUID, List<UUID>>();
        HashSet<UUID> visited = new HashSet<UUID>();
        HashSet<UUID> enqueued = new HashSet<UUID>();
        paths.put(startNode, List.of(startNode));
        deque.addLast(startNode);
        enqueued.add(startNode);
        while (!deque.isEmpty()) {
            UUID currentNode = pollFromDeque.get();
            visited.add(currentNode);
            List currentPath = (List)paths.get(currentNode);
            Set inNeighbors = adjacencyMap.getOrDefault(currentNode, Collections.emptySet());
            Set outNeighbors = invertedAdjacencyMap.getOrDefault(currentNode, Collections.emptySet());
            LinkedHashSet neighbors = new LinkedHashSet(inNeighbors);
            neighbors.addAll(outNeighbors);
            for (UUID neighbor : neighbors) {
                if (visited.contains(neighbor) || enqueued.contains(neighbor)) continue;
                deque.addLast(neighbor);
                enqueued.add(neighbor);
                ArrayList<UUID> path = new ArrayList<UUID>(currentPath);
                path.add(neighbor);
                paths.put(neighbor, path);
            }
        }
        return new GraphSearchResult(visited, paths);
    }

    private GraphSearchResult bfs(Map<UUID, Set<UUID>> adjacencyList, Map<UUID, Set<UUID>> invertedAdjacencyList, UUID startNode) {
        LinkedList deque = new LinkedList();
        return this.graphSearch(adjacencyList, invertedAdjacencyList, startNode, deque, deque::pollFirst);
    }

    private GraphSearchResult dfs(Map<UUID, Set<UUID>> adjacencyList, Map<UUID, Set<UUID>> invertedAdjacencyList, UUID startNode) {
        LinkedList deque = new LinkedList();
        return this.graphSearch(adjacencyList, invertedAdjacencyList, startNode, deque, deque::pollLast);
    }
}

