/*
 * Decompiled with CFR 0.152.
 */
package org.thingsboard.server.service.system;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Charsets;
import com.google.common.hash.Hashing;
import com.google.common.io.Resources;
import jakarta.annotation.PostConstruct;
import java.beans.ConstructorProperties;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;
import org.thingsboard.common.util.JacksonUtil;
import org.thingsboard.common.util.ThingsBoardThreadFactory;
import org.thingsboard.server.common.data.id.TenantId;
import org.thingsboard.server.common.data.widget.WidgetTypeDetails;
import org.thingsboard.server.dao.widget.WidgetTypeService;
import org.thingsboard.server.queue.util.TbCoreComponent;
import org.thingsboard.server.service.install.DatabaseSchemaSettingsService;
import org.thingsboard.server.service.install.InstallScripts;
import org.thingsboard.server.service.install.update.DefaultDataUpdateService;
import org.thingsboard.server.service.system.SystemPatchApplier;

@Component
@TbCoreComponent
public class SystemPatchApplier {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(SystemPatchApplier.class);
    private static final String SCHEMA_VIEWS_SQL = "sql/schema-views.sql";
    private static final long ADVISORY_LOCK_ID = 7536891047216478431L;
    private final JdbcTemplate jdbcTemplate;
    private final InstallScripts installScripts;
    private final DatabaseSchemaSettingsService schemaSettingsService;
    private final WidgetTypeService widgetTypeService;

    @PostConstruct
    private void init() {
        ExecutorService executor = Executors.newSingleThreadExecutor((ThreadFactory)ThingsBoardThreadFactory.forName((String)"system-patch-applier"));
        executor.submit(() -> {
            try {
                this.applyPatchIfNeeded();
            }
            catch (Exception e) {
                log.error("Failed to apply system data patch updates", (Throwable)e);
            }
            finally {
                executor.shutdown();
            }
        });
    }

    private void applyPatchIfNeeded() {
        boolean skipVersionCheck = DefaultDataUpdateService.getEnv((String)"SKIP_PATCH_VERSION_CHECK", (boolean)false);
        if (!skipVersionCheck && !this.isVersionChanged()) {
            return;
        }
        if (!this.acquireAdvisoryLock()) {
            log.trace("Could not acquire advisory lock. Another node is processing patch updates.");
            return;
        }
        try {
            this.updateSqlViews();
            log.info("Updated sql database views");
            int updated = this.updateWidgetTypes();
            log.info("Updated {} widget types", (Object)updated);
            this.schemaSettingsService.updateSchemaVersion();
            log.info("System data patch update completed successfully");
        }
        finally {
            this.releaseAdvisoryLock();
        }
    }

    private boolean isVersionChanged() {
        String packageVersion = this.schemaSettingsService.getPackageSchemaVersion();
        String dbVersion = this.schemaSettingsService.getDbSchemaVersion();
        log.trace("Package version: {}, DB schema version: {}", (Object)packageVersion, (Object)dbVersion);
        VersionInfo packageVersionInfo = this.parseVersion(packageVersion);
        VersionInfo dbVersionInfo = this.parseVersion(dbVersion);
        if (packageVersionInfo == null || dbVersionInfo == null) {
            log.warn("Unable to parse versions. Package: {}, DB: {}", (Object)packageVersion, (Object)dbVersion);
            return false;
        }
        if (!this.isPatchVersionChanged(packageVersionInfo, dbVersionInfo)) {
            return false;
        }
        log.info("Patch version increased from {} to {}. Starting system data update.", (Object)dbVersion, (Object)packageVersion);
        return true;
    }

    private boolean isPatchVersionChanged(VersionInfo packageVersion, VersionInfo dbVersion) {
        return packageVersion.major == dbVersion.major && packageVersion.minor == dbVersion.minor && packageVersion.maintenance == dbVersion.maintenance && packageVersion.patch > dbVersion.patch;
    }

    private void updateSqlViews() {
        try {
            URL schemaViewsUrl = Resources.getResource((String)SCHEMA_VIEWS_SQL);
            String sql = Resources.toString((URL)schemaViewsUrl, (Charset)Charsets.UTF_8);
            this.jdbcTemplate.execute(sql);
        }
        catch (IOException e) {
            throw new RuntimeException("Unable to update database views from schema-views.sql", e);
        }
    }

    private int updateWidgetTypes() {
        AtomicInteger updated = new AtomicInteger();
        Path widgetTypesDir = this.installScripts.getWidgetTypesDir();
        if (!Files.exists(widgetTypesDir, new LinkOption[0])) {
            log.trace("Widget types directory does not exist: {}", (Object)widgetTypesDir);
            return 0;
        }
        try (Stream<Path> dirStream = this.listDir(widgetTypesDir).filter(path -> path.toString().endsWith(".json"));){
            dirStream.forEach(path -> {
                try {
                    if (this.updateWidgetTypeFromFile(path)) {
                        updated.incrementAndGet();
                    }
                }
                catch (Exception e) {
                    log.error("Unable to update widget type from json: [{}]", (Object)path.toString());
                    throw new RuntimeException("Unable to update widget type from json", e);
                }
            });
        }
        return updated.get();
    }

    private boolean updateWidgetTypeFromFile(Path filePath) {
        JsonNode json = JacksonUtil.toJsonNode((File)filePath.toFile());
        WidgetTypeDetails fileWidgetType = (WidgetTypeDetails)JacksonUtil.treeToValue((JsonNode)json, WidgetTypeDetails.class);
        String fqn = fileWidgetType.getFqn();
        WidgetTypeDetails existingWidgetType = this.widgetTypeService.findWidgetTypeDetailsByTenantIdAndFqn(TenantId.SYS_TENANT_ID, fqn);
        if (existingWidgetType == null) {
            throw new RuntimeException("Widget type not found: " + fqn);
        }
        if (this.isWidgetTypeChanged(existingWidgetType, fileWidgetType)) {
            existingWidgetType.setDescription(fileWidgetType.getDescription());
            existingWidgetType.setName(fileWidgetType.getName());
            existingWidgetType.setDescriptor(fileWidgetType.getDescriptor());
            this.widgetTypeService.saveWidgetType(existingWidgetType);
            log.trace("Updated widget type: {}", (Object)fqn);
            return true;
        }
        log.trace("Widget type unchanged: {}", (Object)fqn);
        return false;
    }

    private boolean isWidgetTypeChanged(WidgetTypeDetails existing, WidgetTypeDetails file) {
        if (!this.isDescriptorEqual(existing.getDescriptor(), file.getDescriptor())) {
            return true;
        }
        if (!Objects.equals(existing.getName(), file.getName())) {
            return true;
        }
        return !Objects.equals(existing.getDescription(), file.getDescription());
    }

    private boolean isDescriptorEqual(JsonNode desc1, JsonNode desc2) {
        if (desc1 == null && desc2 == null) {
            return true;
        }
        if (desc1 == null || desc2 == null) {
            return false;
        }
        try {
            String hash1 = this.computeChecksum(desc1);
            String hash2 = this.computeChecksum(desc2);
            return Objects.equals(hash1, hash2);
        }
        catch (Exception e) {
            log.warn("Failed to compare descriptors using checksum, falling back to equals", (Throwable)e);
            return desc1.equals((Object)desc2);
        }
    }

    private String computeChecksum(JsonNode node) {
        String canonicalString = JacksonUtil.toCanonicalString((Object)node);
        if (canonicalString == null) {
            return null;
        }
        return Hashing.sha256().hashBytes(canonicalString.getBytes()).toString();
    }

    private boolean acquireAdvisoryLock() {
        try {
            Boolean acquired = (Boolean)this.jdbcTemplate.queryForObject("SELECT pg_try_advisory_lock(?)", Boolean.class, new Object[]{7536891047216478431L});
            if (Boolean.TRUE.equals(acquired)) {
                log.trace("Acquired advisory lock");
                return true;
            }
            return false;
        }
        catch (Exception e) {
            log.error("Failed to acquire advisory lock", (Throwable)e);
            return false;
        }
    }

    private void releaseAdvisoryLock() {
        try {
            this.jdbcTemplate.queryForObject("SELECT pg_advisory_unlock(?)", Boolean.class, new Object[]{7536891047216478431L});
            log.debug("Released advisory lock");
        }
        catch (Exception e) {
            log.error("Failed to release advisory lock", (Throwable)e);
        }
    }

    private VersionInfo parseVersion(String version) {
        try {
            String[] parts = version.split("\\.");
            int major = Integer.parseInt(parts[0]);
            int minor = parts.length > 1 ? Integer.parseInt(parts[1]) : 0;
            int maintenance = parts.length > 2 ? Integer.parseInt(parts[2]) : 0;
            int patch = parts.length > 3 ? Integer.parseInt(parts[3]) : 0;
            return new VersionInfo(major, minor, maintenance, patch);
        }
        catch (Exception e) {
            log.error("Failed to parse version: {}", (Object)version, (Object)e);
            return null;
        }
    }

    private Stream<Path> listDir(Path dir) {
        try {
            return Files.list(dir);
        }
        catch (NoSuchFileException e) {
            return Stream.empty();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @ConstructorProperties(value={"jdbcTemplate", "installScripts", "schemaSettingsService", "widgetTypeService"})
    @Generated
    public SystemPatchApplier(JdbcTemplate jdbcTemplate, InstallScripts installScripts, DatabaseSchemaSettingsService schemaSettingsService, WidgetTypeService widgetTypeService) {
        this.jdbcTemplate = jdbcTemplate;
        this.installScripts = installScripts;
        this.schemaSettingsService = schemaSettingsService;
        this.widgetTypeService = widgetTypeService;
    }
}

